export  {
    ProcessTimestamp,
    ProcessleftSecond,
    GetUrlParams,
    SetElementStyle,
    DoEveryTime,
    CopyTxt,
    CalcHourInterval,
    GetDayIndex,
    WaitTime,
    OnVisibilityChange,
    EaseNum,
}

/**
 * 监听页面可见性变化
 * @param {Function} onVisible 页面变可见回调
 * @param {Function} onHidden 页面变隐藏回调
 */
function OnVisibilityChange(onVisible, onHidden) {
    document.addEventListener( "visibilitychange" , () => {
        document.visibilityState == "visible" && onVisible?.();
        document.visibilityState == "hidden" && onHidden?.();
    });
}

/**
 * 等待一段时间
 * @param {number} time 等待的时间ms
 */
function WaitTime(time) {
    return new Promise(resolve => setTimeout(resolve, time))
}

/**
 * 计算两个时间戳之间间隔的整点数
 * @param {number} startTimestamp 
 * @param {number} endTimestamp 
 * @returns {number} hours
 */
function CalcHourInterval(startTimestamp, endTimestamp) {
    if (startTimestamp > endTimestamp) {
        console.warn('CalcHourInterval','入参有误！！！ startTimestamp 应该比 endTimestamp 小');
        return 0;
    }
    let hours = Math.floor((endTimestamp - startTimestamp) / 3600000);
    if (endTimestamp % 3600000 < startTimestamp % 3600000) {
        hours += 1;
    }
    return hours
}

/**
 * 复制字符串到剪贴板
 * @param {string} txt 要复制到剪贴板的字符串
 * @param {Function} [success] 成功回调
 * @param {Function} [fail] 失败回调
 */
function CopyTxt(txt, success, fail){
    console.log('尝试复制', txt);
    var textarea = document.createElement('textarea');
    textarea.readOnly = 'readonly'
    textarea.style.position = 'absolute'
    textarea.style.left = '100%'
    textarea.value = txt
    document.body.appendChild(textarea);
    textarea.select();
    // textarea.setSelectionRange(0, textarea.value.length)
    var result = document.execCommand('copy')
    document.body.removeChild(textarea)
    if (result) {
        console.log('复制成功');
        success && success()
    } else {
        console.log('复制失败');
        fail && fail()
    }
};

/**
 * EaseNum的回调函数
 * @callback OnEase
 * @param {number} curNum 当前数字
 * @param {Function} stopEase 停止缓动的方法
 */
/**
 * 缓动数字（整数）
 * @param {number} startNum 开始数
 * @param {number} endNum 结束数
 * @param {number} time 缓动时间
 * @param {OnEase} onEase 缓动每步回调
 * @returns {Function} 停止缓动方法
 */
 function EaseNum(startNum, endNum, time, onEase) {
    // console.log('EaseNum', startNum, endNum, time);
    let stopfn = ()=>{};
    const stopEase = () => stopfn();
    if (time == 0 || startNum == endNum) {
        onEase(endNum, stopEase);
    } else {
        let interval = 50; // 基础变化间隔时间
        let perNum = (endNum - startNum) / (time / interval); // 单次增加量
        if (perNum > 0) {
            if (perNum < 1) {
                perNum = 1;
                interval = Math.floor(time / (endNum - startNum));
            } else {
                interval = Math.floor(interval * Math.ceil(perNum) / perNum);
                perNum = Math.ceil(perNum);
            }
        }
        if (perNum < 0) {
            if (perNum > -1) {
                perNum = -1;
                interval = Math.floor(time / (startNum - endNum));
            } else {
                interval = Math.floor(interval * Math.floor(perNum) / perNum);
                perNum = Math.floor(perNum);
            }
        }
        // console.log('perNum', perNum, 'interval', interval);
        let curNum = startNum;
        stopfn = DoEveryTime(interval, (dt, stop) => {
            stopfn = stop;
            if ((perNum > 0 && (curNum >= endNum)) || (perNum < 0 && (curNum <= endNum))) {
                curNum = endNum;
                stop();
            } else {
                curNum += perNum;
            }
            onEase(curNum, stopEase);
        })
    }
    return stopEase
}

/**
 * DoEveryTime的回调函数
 * @callback everyTimeCallback
 * @param {number} dt 实际间隔时间
 * @param {function} stopContinue 停止继续执行的方法
 */
/**
 * 每time毫秒执行一次fn
 * @param {number} time 时间间隔，毫秒
 * @param {everyTimeCallback} fn 需要间隔执行的方法
 * @returns 返回停止第一次执行的方法
 */
function DoEveryTime(time, fn) {
    let expectedTime = Date.now();
    let preTime = expectedTime;
    let startTimer = setTimeout(doIt, time);
    function doIt() {
        expectedTime = expectedTime + time;
        let nowTime = Date.now();
        let offset = expectedTime - nowTime;      
        let nextTime = time + offset;
        if (nextTime < 0) nextTime = 0;
        let dt = nowTime - preTime;
        preTime = nowTime
        let continueTimer = setTimeout(doIt, nextTime);  
        let stopContinue = ()=>{continueTimer && clearTimeout(continueTimer)};
        fn(dt, stopContinue);
    }
    let stopStart = ()=>{startTimer && clearTimeout(startTimer)}
    return stopStart
}

/**
 * 设置元素style
 * @param {object} option 配置
 * @param {HTMLElement} option.element HTMLElement
 * @param {object} option.style element.style
 * @param {Function} [option.onTransitionEnd] transitionend回调
 * @param {Function} [option.onAnimationEnd] animationend回调
 */
function SetElementStyle({element, style, onTransitionEnd, onAnimationEnd}) {
    const { transition, animation } = style;
    if(transition !== undefined) element.style.transition = transition;
    if(animation !== undefined) element.style.animation = animation;
    setTimeout(()=>{
        Object.keys(style).forEach(v => {
            if (v != 'transition' && v != 'animation') {
                element.style[v] = style[v];
            }
        });
    })
    if (onTransitionEnd) {
        element.removeEventListener('transitionend', onTransitionEnd);
        element.addEventListener('transitionend', onTransitionEnd);
    }
    if (onAnimationEnd) {
        element.removeEventListener('animationend', onAnimationEnd);
        element.addEventListener('animationend', onAnimationEnd);
    }
}

/**
 * 获取url中的请求参数
 * @param {string} url 链接
 * @returns 返回参数键值对object
 */
function GetUrlParams(url){
    var paramObject = {};
    var rawParamStr = url.split('?')[1];
    if (rawParamStr) {
        var rawKeyValues = rawParamStr.split('&');
        rawKeyValues.forEach((v,i)=>{
            var key = v.split('=')[0];
            var value = v.split('=')[1];
            paramObject[key] = decodeURIComponent(value);
        })  
    } 
    return paramObject; 
};

/**
 * 加工时间戳
 * @param {number} timestamp 时间戳
 * @param {string} type 输出类型，自定义
 */
 function ProcessTimestamp(timestamp, type) {
    let time =  new Date(+timestamp);
    let yearStr = time.getFullYear();
    let month = time.getMonth() + 1;
    let monthStr = month > 9 ? month : '0' + month;
    let date = time.getDate();
    let dateStr = date > 9 ? date : '0' + date;
    let hour = time.getHours();
    let hourStr = hour > 9 ? hour : '0' + hour;
    let minute = time.getMinutes();
    let minuteStr = minute > 9 ? minute : '0' + minute;
    let second = time.getSeconds();
    let secondStr = second > 9 ? second : '0' + second;
    switch (type) {
        case '1':
            return `${yearStr}-${monthStr}-${dateStr} ${hourStr}:${minuteStr}:${secondStr}`
        case '2':
            return `${yearStr}.${monthStr}.${dateStr} ${hourStr}:${minuteStr}`
    }
}

/**
 * 加工秒数，输出HH:MM:SS
 * @param {number} leftSecond 剩余秒数
 */
function ProcessleftSecond(leftSecond) {
	let hour = Math.floor(leftSecond / 3600);
	let hourStr = hour.toString().length == 1 ? `0${hour}` : `${hour}`;
	let minute = Math.floor((leftSecond % 3600) / 60);
	let minuteStr = minute.toString().length == 1 ? `0${minute}` : `${minute}`;
	let second = Math.ceil(leftSecond - 3600*hour - 60*minute);//不足1秒按1秒计
	let secondStr = second.toString().length == 1 ? `0${second}` : `${second}`;
	return `${hourStr}:${minuteStr}:${secondStr}`
}

/**
 * 获取星期索引：0到6对应星期一到星期日
 * @param {string|number} time 标准日期格式或时间戳
 */
function GetDayIndex(time) {
    if (typeof time == 'string') {
        time = time.replace(/-/g, '/');
    }
    let index = new Date(time).getDay() - 1;
    if(index == -1) index = 6;
    return index;
}