export { InputController }

/**
 * 输入控制类，用于节流（可防连点）和防抖控制
 */
class InputController {
    /**
     * 节流控制方法
     * @param {string} sign 控制的标记，不同控制用不同的sign，建议写方法名
     * @param {number} [time] 控制的时间，默认1500ms，为0则一直控制，直到通过相应sign来解除控制
     * @returns {boolean} 要控制 = true | 不控制 = false
     * @example 
     * //防连点：
     * onClick = {() => throttle_control('doSomething1') || doSomething1()}
     * //手动开关：
     * onClick = {() => throttle_control('doSomething2', 0) || doSomething2()}
     * //第一次点击可以正常走到doSomething2()，后续点击走不到，通过执行下面代码来解除控制：
     * throttle_control['doSomething2'] = false;
     */
    throttle_control = (sign, time = 1500) => {
        if (this.throttle_control[sign]) {
            return true;
        } else {
            this.throttle_control[sign] = true;
            if (time) {
                setTimeout(() => {
                    this.throttle_control[sign] = false;
                }, time);
            }
            return false;
        }
    }

    /**
     * 节流改造方法
     * @param {function} fn 要节流的方法
     * @param {number} time 间隔时间，默认1500ms
     * @returns {function} 带有节流功能的方法，由fn改造而来
     * @example 
     * //把目标方法doSomething1改造成自带节流功能的方法doSomething2
     * doSomething2 = throttle_reform(doSomething1);
     */
    throttle_reform = (fn, time = 1500) => {
        let sign = false;
        return function throttleFn() {
            if (!sign) {
                sign = true;
                fn();
                setTimeout(() => {
                    sign = false;
                }, time);
            }
        }
    }

    /**
     * 防抖控制方法
     * @param {string} sign 控制的标记
     * @param {number} [time] 控制的时间，默认为500ms
     * @example
     * //input输入框onChange回调防抖：
     * onChange = {async () => await debounce_control('doSomething1') || doSomething1()}
     */
    debounce_control = (sign, time = 500) => {
        let isRelease = false;//记录promise是否释放
        return new Promise((resolve)=>{
            this.debounce_control[sign] && clearTimeout(this.debounce_control[sign]);
            this.debounce_control[sign] = setTimeout(() => {
                resolve(false);
            }, time);
            setTimeout(() => {
                if (!isRelease) {//未释放则释放
                    resolve(true);
                }
            }, time + 20);
        })
    }

    /**
     * 防抖改造方法
     * @param {function} fn 要防抖的方法
     * @param {number} time 间隔时间，默认500ms
     * @returns {function} 带有防抖功能的方法，由fn改造而来
     */
    debounce_reform = (fn, time = 500) => {
        let timer = null
        return function () {
            timer && clearTimeout(timer)
            timer = setTimeout(() => {
                fn();
            }, time)
        }
    }
}