'use strict';
import { Global } from '@src/global';
import * as FYGE from 'fyge';
import React, { useEffect, useState } from 'react';
import './comp.less';
export {
  Comp_fyge_canvas,
  addClick,
  createContainer,
  createTxt,
  createImg,
  createSvga,
  createSpine,
  createLottie,
}

/**
 * 使用fyge引擎操作的canvas组件
 * @param {object} props
 * @param {Function} [props.onStageReady] stage准备好了回调，回调入参为stage实例
 */
function Comp_fyge_canvas({ onStageReady }) {
  const [that] = useState({});
  // 生成stage
  useEffect(() => {
    console.warn('Comp_fyge_canvas init');
    const canvas = that.canvas;
    //建舞台
    const stage = new FYGE.Stage(
      canvas,//canvas画布
      750,//设计宽度，按设计搞给的就行
      1624,//设计高度
      canvas.getBoundingClientRect().width,//显示宽度，css宽度
      canvas.getBoundingClientRect().height,//显示高度，css高度
      FYGE.RENDERER_TYPE.WEBGL,//2d上下文
      false,//舞台是否居中，false不居中左上置顶，true居中小于1624或750的视窗会上下或左右平均裁切
      false,//是否定高，默认否（定宽）
      window.devicePixelRatio || 1
    );
    //鼠标事件
    const mouseEvent = stage.onMouseEvent.bind(stage);
    stage.webMouseEventPreventDefault = true;
    stage.webMouseEventStopPropagation = true;
    canvas.addEventListener("touchstart", mouseEvent, false);
    canvas.addEventListener('touchmove', mouseEvent, false);
    canvas.addEventListener('touchend', mouseEvent, false);
    //stage初始化事件
    stage.addEventListener(FYGE.Event.INIT_STAGE, () => onStageReady?.(stage));
    //每帧循环起来
    const loop = () => {
      FYGE.Tween.flush()
      stage.flush();
      that.requestID = window.requestAnimationFrame(loop);
    }
    loop();
    return () => {
      //Tween都移除，
      FYGE.Tween.removeAllTweens()
      //计时器取消
      window.cancelAnimationFrame(that.requestID);
      //舞台销毁
      stage.destroy();
    }
  }, [])

  return (
    <canvas className="fyge_canvas" ref={ref => that.canvas = ref}></canvas>
  );
}

/**
 * 添加点击事件
 * @param {*} node displayObject
 * @param {Function} onClick 点击回调函数
 * @param {*} data 点击事件携带数据，可通过event.target.__data获取
 */
function addClick(node, onClick, data = null) {
  node.__data = data;
  node.removeEventListener(FYGE.MouseEvent.CLICK, onClick);
  node.addEventListener(FYGE.MouseEvent.CLICK, onClick);
}

/**
 * 创建容器
 * @param {object} option 
 */
function createContainer({
  x = 0,
  y = 0,
  width = 0,
  height = 0,
  color,
  colorOpacity = 0.3,
  visible = true,
  onCreated,
}) {
  const container = new FYGE.Container();
  container.position.set(x, y);
  container.width = width;
  container.height = height;
  container.visible = visible;
  if (color) {
    container.addChild(new FYGE.Graphics())
      .beginFill(color, colorOpacity)
      .drawRect(0, 0, width, height)
      .endFill();
  }
  onCreated?.(container);
  return container
}

/**
 * 创建文字
 * @param {object} option 
 */
function createTxt({
  text = '', //显示的文字内容
  x = 0,
  y = 0,
  color = "#000000", //文本颜色
  size = 16, //文字大小
  width, //文字的宽度
  height, //文字的高度
  align = 'left', //文字的对齐方式
  strokeColor, //文本的描边颜色
  stroke, //描边的宽度
  visible = true,
  onCreated
}) {
  const txt = new FYGE.TextField();
  txt.text = text;
  txt.position.set(x, y);
  txt.fillColor = color;
  txt.size = size;
  width && (txt.textWidth = width);
  height && (txt.textHeight = height);
  txt.textAlign = align;
  strokeColor && (txt.strokeColor = strokeColor);
  stroke && (txt.stroke = stroke);
  txt.visible = visible;
  onCreated?.(txt);
  return txt
}

/**
 * 创建图片
 * @param {object} option 
 */
function createImg({
  path,
  x = 0,
  y = 0,
  scaleX = 1,
  scaleY = 1,
  visible = true,
  onCreated
}) {
  const img = new FYGE.Sprite(FYGE.Texture.fromImage(Global.assetData[path]));
  img.position.set(x, y);
  img.scaleX = scaleX;
  img.scaleY = scaleY;
  img.visible = visible;
  onCreated?.(img);
  return img
}

/**
 * 创建svga
 * @param {object} option 
 */
function createSvga({
  path,
  x = 0,
  y = 0,
  scaleX = 1,
  scaleY = 1,
  autoPlay = true,
  loop = 0,
  onEnd,
  mouseEnable = false,
  visible = true,
  onCreated,
}) {
  const svga = new FYGE.MovieClip(Global.assetData[path]); //创建svga对象
  svga.visible = visible;
  svga.position.set(x, y); //设置svga位置
  svga.scaleX = scaleX;
  svga.scaleY = scaleY;
  if (autoPlay) {
    svga.startAniRange(0, svga.totalFrames, loop, () => onEnd?.());
  } else {
    svga.stop();
  }
  svga.mouseEnable = mouseEnable;
  svga.mouseChildren = mouseEnable;
  onCreated?.(svga);
  return svga
}

/**
 * 创建spine
 * @param {object} option 
 */
function createSpine({
  path,
  x = 0,
  y = 0,
  scaleX = 1,
  scaleY = 1,
  autoPlay = true,
  skinName,
  aniName,
  loop = 0,
  onEnd,
  mouseEnable = false,
  visible = true,
  onCreated,
}) {
  const spine = new FYGE.Spine(Global.assetData[path])
  spine.visible = visible;
  spine.position.set(x, y);
  spine.scaleX = scaleX;
  spine.scaleY = -scaleY;
  skinName && spine.setSkin(skinName);
  console.log("skinName", spine);
  if (autoPlay && aniName) {
    spine.animationManager.showAni(aniName, loop, () => onEnd?.());
  }
  spine.mouseEnable = mouseEnable;
  spine.mouseChildren = mouseEnable;
  onCreated?.(spine);
  return spine
}

/**
 * 创建lottie
 * @param {object} option 
 */
 function createLottie({
  path,
  x = 0,
  y = 0,
  scaleX = 1,
  scaleY = 1,
  autoPlay = true,
  loop = 0,
  onEnd,
  mouseEnable = false,
  visible = true,
  onCreated,
}) {
  const lottie = new FYGE.Lottie(Global.assetData[path])
  lottie.visible = visible;
  lottie.position.set(x, y);
  lottie.scaleX = scaleX;
  lottie.scaleY = scaleY;
  if (autoPlay) {
    lottie.play(loop, () => onEnd?.());
  }
  lottie.mouseEnable = mouseEnable;
  lottie.mouseChildren = mouseEnable;
  onCreated?.(lottie);
  return lottie
}
