import React from 'react';

import { observer } from 'mobx-react';

import common from 'src/lib/common';
import store from '../../../store';

import {
  Icon,
  Upload,
  Modal
} from 'antd';

const url = '/upload/uploadImg';
@observer
class PicturesWall extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      previewVisible: false,
      previewImage: props.value
    };
  }
  // 上传前进行检查
  handleBeforeUpload = (file, fileList) => {
    const { width, height, fileLimit } = this.props;
    const imgTypeSize = {
      width: parseInt(width, 10) || undefined,
      height: parseInt(height, 10) || undefined
    };
    // 检查文件是否是图片
    if (!this.checkIsImage(file)) {
      common.handleError('上传的必须是jpg, jpeg, png类型的图片');
      return false;
    }
    // 检查文件大小是否超过限制
    if (!this.checkFileLimit(file)) {
      const fileLimitText = fileLimit >= (1 * 1024 * 1024) ? `${(fileLimit / 1024 / 1024)}MB` : `${(fileLimit / 1024)}KB`;
      common.handleError(`图片必须小于${fileLimitText}`);
      return false;
    }
    // 检查文件尺寸是否超过限制
    return this.checkImageSize(file, imgTypeSize);
  }
  handleChange = (file) => {
    const { updatePhoto } = store;
    let fileList = file.fileList;
    if (file.file.status === 'done') {
      const res = file.file.response;
      if (res['code'] === '0') {
        common.handleSuccess('上传图片成功');
        updatePhoto(fileList);
        return;
      } else {
        common.handleError(res['desc']);
      }
    } else if (file.file.status === 'error') {
      common.handleError('上传图片请求出错，请稍候重试');
    }
    updatePhoto(fileList);
  }
  // 处理预览框取消事件
  handleCancel = () => this.setState({ previewVisible: false })
  // 处理预览按钮点击事件
  handlePreview = (file) => {
    this.setState({
      previewImage: file.url || file.thumbUrl,
      previewVisible: true
    });
  }
  handleRemove = (file) => {
    const { deletePhoto } = store;
    let id;
    if (!file.response) {
      id = file.uid;
    } else {
      id = 1;
    }
    let params = {
      id: -id
    };
    return new Promise((resolve, reject) => {
      Modal.confirm({
        title: '提示',
        content: '确认修改?',
        okText: '确定',
        okType: 'danger',
        cancelText: '取消',
        onOk() {
          deletePhoto(params.id).then(resolve());
        },
        onCancel() {
          console.log('you chose close!');
          resolve(false);
        }
      });
    });
  }
  // 校验图片的大小是否合规
  checkImageSize = (file, imgTypeSize) => {
    return new Promise(function(resolve, reject) {
      function check(width, height) {
        if (
          (width === imgTypeSize.width || !imgTypeSize.width) &&
          (height === imgTypeSize.height || !imgTypeSize.height)
        ) {
          resolve(file);
        } else {
          common.handleError(
            '图片大小不对，要求尺寸为' +
              (imgTypeSize.width || '宽度不限') +
              '*' +
              (imgTypeSize.height || '高度不限') +
              ' 像素'
          );
        }
      }
      // 读取图片数据
      if (typeof window.FileReader === 'function') {
        // 读取图片数据
        const reader = new window.FileReader();
        reader.onload = function(e) {
          const data = e.target.result;
          // 加载图片获取图片真实宽度和高度
          const image = new window.Image();
          image.onload = function() {
            const width = image.width;
            const height = image.height;
            check(width, height);
          };
          image.src = data;
        };
        reader.readAsDataURL(file);
      } else {
        common.handleError('你的浏览器不支持图片大小判断，请更换现代浏览器，例如Chrome');
      }
    });
  }
  // 限制文件格式为图片
  checkIsImage = (file) => {
    const fileName = file.name;
    return !!fileName.match(/.jpg|.jpeg|.png/);
  }
  // 限制文件大小
  checkFileLimit = (file) => {
    const { fileLimit } = this.props;
    if (!fileLimit) {
      return true;
    }
    return fileLimit > file.size;
  }
  render() {
    const { uploadButtonText = '上传图片' } = this.props;
    const { previewVisible, previewImage } = this.state;
    const uploadButton = (
      <div>
        <Icon type="plus" />
        <div className="ant-upload-text">{uploadButtonText}</div>
      </div>
    );
    const { fileList } = store;
    let imgList = [];
    fileList && fileList.forEach(item => {
      if (item.status) {
        imgList.push(item);
      } else {
        let obj = {};
        obj.uid = -item.id;
        obj.name = item.imgUrl;
        obj.url = item.imgUrl;
        obj.status = 'done';
        imgList.push(obj);
      }
    });
    return (
      <div className="clearfix">
        <Upload
          action={url}
          listType="picture-card"
          beforeUpload={this.handleBeforeUpload}
          fileList={imgList}
          onPreview={this.handlePreview}
          onChange={this.handleChange}
          onRemove={this.handleRemove}
        >
          {fileList.length >= 3 ? null : uploadButton}
        </Upload>
        <Modal visible={previewVisible} footer={null} onCancel={this.handleCancel}>
          <img alt="图片" src={previewImage || (fileList.length && fileList.url)} />
        </Modal>
      </div>
    );
  }
}

export default PicturesWall;
