package com.duiba.synredis.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class HadoopRedisUtil {
    private static final Logger logger = LoggerFactory.getLogger(HadoopRedisUtil.class);

    private static JedisPool jedisPool = null;
    //volatile 禁止jvm进行指令重排
    private volatile static HadoopRedisUtil instance = null;

    private HadoopRedisUtil(int maxActive) {
        //加载配置文件
        PropertyUtil instance = PropertyUtil.getInstance();
        if (maxActive == 0) {
            maxActive = instance.getPropertyForInt("redis.max_active");
        }

        //创建redis连接池
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxIdle(instance.getPropertyForInt("redis.max_idle"));
        config.setMinIdle(instance.getPropertyForInt("redis.min_idle"));
        config.setMaxWaitMillis(instance.getPropertyForLong("redis.max_wait"));
        config.setMaxTotal(maxActive);
        config.setTestOnBorrow(instance.getPropertyForBoolean("redis.test_on_borrow"));
        config.setTestWhileIdle(true);
        config.setTimeBetweenEvictionRunsMillis(instance.getPropertyForLong("redis.test_while_idle"));
        jedisPool = new JedisPool(config, instance.getProperty("hadoop.redis.host"), instance.getPropertyForInt("redis.port"),
                instance.getPropertyForInt("redis.timeout"), instance.getProperty("hadoop.redis.auth"));
    }

    public static HadoopRedisUtil getInstance() {
        return getInstance(0);
    }

    public static HadoopRedisUtil getInstance(int maxActive) {
        if (instance == null) {
            init(maxActive);
        }
        return instance;
    }

    private synchronized static void init(int maxActive) {
        if (instance == null) {
            instance = new HadoopRedisUtil(maxActive);
        }
    }

    /**
     * 获取Jedis实例
     *
     * @return redis连接
     */
    private Jedis getJedis() {
        return jedisPool.getResource();
    }

    /**
     * 释放jedis资源
     *
     * @param jedis redis连接
     */
    private void returnResource(Jedis jedis) {
        try {
            if (jedis != null) {
                jedis.close();
            }
        } catch (Exception e) {
            logger.error("returnResource error.", e);
        }
    }

    /**
     * 返回hash中指定存储位置的值
     *
     * @param key   键值
     * @param fieid 存储的名字
     * @return 存储对应的值
     */
    public String hget(String key, String fieid) {
        Jedis jedis = getJedis();
        try {
            return jedis.hget(key, fieid);
        } catch (Exception e) {
            logger.error("redis hget error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 添加对应关系，如果对应关系已存在，则覆盖
     *
     * @param key   键值
     * @param field map中的key
     * @param value map中的value
     */
    public void hset(String key, String field, String value) {
        Jedis jedis = getJedis();
        try {
            jedis.hset(key, field, value);
        } catch (Exception e) {
            logger.error("redis hmset error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 添加对应关系，如果对应关系已存在，则覆盖
     *
     * @param key 键值
     * @param map 存入redis中的map字段
     */
    public void hmset(String key, Map<String, String> map) {
        Jedis jedis = getJedis();
        try {
            jedis.hmset(key, map);
        } catch (Exception e) {
            logger.error("redis hmset error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 以Map的形式返回hash中的存储和值
     *
     * @param key 键值
     * @return map对象
     */
    public Map<String, String> hgetAll(String key) {
        Jedis jedis = getJedis();
        try {
            return jedis.hgetAll(key);
        } catch (Exception e) {
            logger.error("redis hgetAll error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 计数器 指定key field 累计
     *
     * @param key 键值
     * @return map对象
     */
    public void hincrBy(String key, String field, Long value) {
        Jedis jedis = getJedis();
        try {
            jedis.hincrBy(key, field, value);
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 计数器 指定key field 累计
     * 并设置生效时间
     *
     * @param key 键值
     * @return map对象
     */
    public void hincrBy(String key, String field, Long value, int seconds) {
        Jedis jedis = getJedis();
        try {
            jedis.hincrBy(key, field, value);
            jedis.expire(key, seconds);
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 计数器 批量计数
     *
     * @param key 键值
     * @return map对象
     */
    public void hincrBy(String key, Map<String, Long> map) {
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            for (Map.Entry<String, Long> entry : map.entrySet()) {
                pipe.hincrBy(key, entry.getKey(), entry.getValue());
            }
            pipe.sync();
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 计数器 批量计数
     *
     * @param key 键值
     * @return map对象
     */
    public void hincrBy(String key, Map<String, Long> map, int seconds) {
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            for (Map.Entry<String, Long> entry : map.entrySet()) {
                pipe.hincrBy(key, entry.getKey(), entry.getValue());
            }
            pipe.expire(key, seconds);
            pipe.sync();
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }


    /**
     * 计数器 批量计数
     *
     * @param redisKeyList redisKey集合
     * @return map对象
     */
    public void hincrBy(List<String> redisKeyList, Map<String, Long> map) {
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            for (String redisKey : redisKeyList) {
                for (String column : map.keySet()) {
                    pipe.hincrBy(redisKey, column, map.get(column));
                }
            }
            pipe.sync();
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 判断key是否存在
     *
     * @param key 键值
     * @return 是否存在 true表示存在
     */
    public Boolean exists(String key) {
        Jedis jedis = getJedis();
        try {
            return jedis.exists(key);
        } catch (Exception e) {
            logger.error("redis exists error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 设置key的过期时间
     *
     * @param key     键值
     * @param seconds 秒
     * @return 设置成功返回 1 否则返回 0
     */
    public Long expire(String key, int seconds) {
        Jedis jedis = getJedis();
        try {
            return jedis.expire(key, seconds);
        } catch (Exception e) {
            logger.error("redis expire error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 设置key的过期时间
     *
     * @param redisKeyList rediskey集合
     * @param seconds      秒
     */
    public void expire(List<String> redisKeyList, int seconds) {
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            for (String redisKey : redisKeyList) {
                pipe.expire(redisKey, seconds);
            }
            pipe.sync();
        } catch (Exception e) {
            logger.error("redis expire error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 通过pipeline 批量插入数据
     *
     * @param map     key,value 键值对
     * @param seconds 过期时间（单位：秒）
     */
    public void pipeLineMset(Map<String, String> map, int seconds) {
        Jedis jedis = getJedis();
        try {
            if (map != null && map.size() > 0) {
                Pipeline pipeline = jedis.pipelined();
                for (String key : map.keySet()) {
                    pipeline.setex(key, seconds, map.get(key));
                }
                pipeline.sync();
            }
        } catch (Exception e) {
            logger.error("redis pipeLineMset error.", e);
        } finally {
            returnResource(jedis);
        }
    }


    /**
     * 批量删除key
     *
     * @param keyList 需要删除的key集合
     */
    public void pipeLineDel(List<String> keyList) {
        Jedis jedis = getJedis();
        try {
            if (keyList != null && keyList.size() > 0) {
                Pipeline pipeline = jedis.pipelined();
                for (String key : keyList) {
                    pipeline.del(key);
                }
                pipeline.sync();
            }
        } catch (Exception e) {
            logger.error("redis pipeLineDel error.", e);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 通过key获取redis中数据
     *
     * @param key 键值
     * @return value
     */
    public String get(String key) {
        Jedis jedis = getJedis();
        try {
            return jedis.get(key);
        } catch (Exception e) {
            logger.error("redis get error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 通过key写入redis中数据
     *
     * @param key   键值
     * @param value 值
     * @return value
     */
    public String set(String key, String value, int expire) {
        Jedis jedis = getJedis();
        try {
            return jedis.setex(key, expire, value);
        } catch (Exception e) {
            logger.error("redis set error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    public Long del(String key) {
        Jedis jedis = getJedis();
        try {
            return jedis.del(key);
        } catch (Exception e) {
            logger.error("redis del error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 批量查询数据
     *
     * @param keyList 指定查询的key集合，一次查询1000条，大量数据分批调用
     * @return
     */
    public Map<String, Map<String, String>> hmget(List<String> keyList) {
        Map<String, Map<String, String>> resultMap = new HashMap<>();
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            Map<String, Response<Map<String, String>>> tmpMap = new HashMap<>();
            for (String key : keyList) {
                tmpMap.put(key, pipe.hgetAll(key));
            }
            pipe.sync();
            for (Map.Entry<String, Response<Map<String, String>>> entry : tmpMap.entrySet()) {
                // 解析map jie结果值
                resultMap.put(entry.getKey(), entry.getValue().get());
            }
        } catch (Exception e) {
            logger.error("redis hmget error.", e);
        } finally {
            returnResource(jedis);
        }
        return resultMap;
    }

    /**
     * 通过key存入值
     *
     * @param key
     * @param value
     * @return
     */
    public String set(String key, String value) {
        Jedis jedis = getJedis();
        try {
            return jedis.set(key, value);
        } catch (Exception e) {
            logger.error("redis get error.", e);
        } finally {
            returnResource(jedis);
        }
        return null;
    }

    /**
     * 批量查询数据
     *
     * @param keyList 指定查询的key集合，一次查询1000条，大量数据分批调用
     * @return
     */
    public Map<String, String> pipeLineMget(List<String> keyList) {
        Map<String, String> resultMap = new HashMap<>();
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            Map<String, Response<String>> tmpMap = new HashMap<>();
            for (String key : keyList) {
                tmpMap.put(key, pipe.get(key));
            }
            pipe.sync();
            for (Map.Entry<String, Response<String>> entry : tmpMap.entrySet()) {
                // 解析map jie结果值
                resultMap.put(entry.getKey(), entry.getValue().get());
            }
        } catch (Exception e) {
            logger.error("redis pipeLineMget error.", e);
        } finally {
            returnResource(jedis);
        }
        return resultMap;
    }

    public void pipeLineHincrBy(String field, Map<String, Long> map, int seconds) {
        Jedis jedis = getJedis();
        try {
            Pipeline pipe = jedis.pipelined();
            for (Map.Entry<String, Long> entry : map.entrySet()) {
                String key = entry.getKey();
                pipe.hincrBy(key, field, entry.getValue());
                pipe.expire(key, seconds);
            }
            pipe.sync();
        } catch (Exception e) {
            logger.error("redis hincrBy error.", e);
        } finally {
            returnResource(jedis);
        }
    }
}
