package com.duiba.synredis.util;

import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.util.Map;

public class ElasticSearchUtil {
    private static final Logger logger = LoggerFactory.getLogger(ElasticSearchUtil.class);
    private volatile static ElasticSearchUtil instance = null;
    private TransportClient client;

    public static ElasticSearchUtil getInstance() {
        if (instance == null) {
            init();
        }
        return instance;
    }

    private synchronized static void init() {
        if (instance == null) {
            instance = new ElasticSearchUtil();
        }
    }

    private ElasticSearchUtil() {
        initClient();
    }

    private synchronized TransportClient initClient() {
        if (client != null) {
            return client;
        }
        try {
            //加载配置文件
            PropertyUtil instance = PropertyUtil.getInstance();
            Settings settings = Settings.settingsBuilder().put("cluster.name", instance.getProperty("es.cluster.name"))
                    .put("client.transport.sniff", true).build();
            String hostsports = instance.getProperty("es.hostsports");
            if (StringUtils.isNotBlank(hostsports)) {
                client = TransportClient.builder().settings(settings).build();
                String[] hostsPortStrs = hostsports.split(",");
                for (String hostport : hostsPortStrs) {
                    String[] strs = hostport.split(":");
                    client.addTransportAddress(new InetSocketTransportAddress(InetAddress.getByName(strs[0]), Integer.valueOf(strs[1])));
                }
            }
            return client;
        } catch (Exception e) {
            logger.error("initClient error", e);
        }
        return null;
    }

    public void batchInsertDoc(String index, String type, Map<String, Map<String, Object>> map) {
        try{
            if (StringUtils.isNotBlank(index) && StringUtils.isNotBlank(type) && null != map) {
                BulkRequestBuilder bulkRequest = client.prepareBulk();
                for (String key : map.keySet()) {
                    bulkRequest.add(client.prepareIndex(index, type,key).setSource(map.get(key)));
                }
                BulkResponse bulkResponse = bulkRequest.execute().actionGet();
                if (bulkResponse.hasFailures()) {
                    logger.error(bulkResponse.buildFailureMessage());
                }
            }
        }
        catch (Exception e){
            logger.error("batchInsertDoc error.", e);
        }
    }


    private void closeClient() {
        try {
            if (client != null) {
                client.close();
            }
        } catch (Exception e) {
            logger.error("closeClient error.", e);
        }
    }

    public static void main(String[] args) throws Exception {

    }

}
