import { Eureka } from 'eureka-js-client';

const address = require('address');

const ip = address.ip() || '127.0.0.1';

interface IParams {
  port: number;
  appName: string;
}

function getParams(options: IParams) {
  const { port, appName } = options;

  const env = process.env.NODE_ENV || 'dev';

  const eurekaHosts = {
    dev: ['http://eureka.duibadev.com.cn'],
    test: ['http://eureka.duibatest.com.cn'],
    autotest: ['http://autotest-eureka.duibatest.com.cn'],
    aliyunTest: ['http://10.170.1.121:1025'],
    pre: ['http://10.30.10.61:1025'],
    prod: ['http://10.10.10.21:1025', 'http://10.20.10.22:1025']
  };

  let msg: string;

  if (!env) {
    msg = '无法获取到NODE_ENV';
  } else if (!eurekaHosts[env]) {
    msg = `NODE_ENV错误，只能为${Object.keys(eurekaHosts).join(',')}`;
  }

  if (msg) {
    console.error(new Error(`Eureka Error，${msg}`));
    process.exit(1);
  }

  return {
    // application instance information
    instance: {
      instanceId: `${ip}:${appName}:${port}`,
      hostName: ip,
      app: appName,
      ipAddr: ip,
      status: 'UP',
      port: { $: port, '@enabled': 'true' },
      securePort: { $: '443', '@enabled': 'false' },
      homePageUrl: `http://${ip}:${port}/`,
      statusPageUrl: `http://${ip}:${port}/info`,
      healthCheckUrl: `http://${ip}:${port}/health`,
      lastDirtyTimestamp: +new Date(),
      vipAddress: appName,
      secureVipAddress: appName,
      dataCenterInfo: {
        '@class': 'com.netflix.appinfo.InstanceInfo$DefaultDataCenterInfo',
        name: 'MyOwn'
      },
      metadata: {
        weight: '100',
        'duiba.service.group.key': process.env.DUIBA_SERVICE_GROUP_KEY,
        zone: process.env.ZONE || 'defaultZone',
        'run.in.docker': process.env.RUN_IN_DOCKER || "false",
        'runInSingleJarMode': process.env.RUN_IN_DOCKER || "false",
        'duiba.perf.scene.id': process.env.DUIBA_PERF_SCENE_ID
      }
    },
    eureka: {
      // eureka server host / port
      serviceUrls: {
        default: eurekaHosts[env].map(el => `${el}/eureka/apps/`)
      }
    }
  };
}

export default function client(options: IParams) {
  const params: any = getParams(options);

  const client = new Eureka(params);

  return client;
}
