const config = require('./config');

const expressHandler = require('./lib/express-handler');
const koaHandler = require('./lib/koa-handler');
const eggHandler = require('./lib/egg-handler');

const NODE_ENV = process.env.NODE_ENV;
const isLocal = NODE_ENV === 'local' || NODE_ENV === undefined;

class NodeDocker {
  constructor(options) {
    this.options = options;
    this.init();
    return this;
  }
  init() {
    this.registerRoute();
    !isLocal && this.onSignals();
  }
  startServer() {
    return new Promise(resolve => {
      const { app, port } = this.options;
    
      this.serverStatus = 'OPENING';
  
      this.server = app.listen(port, () => {
        resolve()
      });
      this.onConnection();
  
      console.log(`Server listen on port ${port}`);
    })
  }
  stopServer(cb) {

    this.serverStatus = 'SHUTDOWN';

    // 取消Eureka注册
    if (global.Eureka) {
      global.Eureka.stop();
    }

    setTimeout(() => {
      //关闭之前，我们需要手动清理连接池中得socket对象
      this.sockets.forEach((socket) => {
        socket.destroy();
      });
      this.server.close(() => {
        cb && cb('SHUTDOWN');
        process.exit(2);
      });
    }, config.delayStop)
  }
  registerRoute() {
    switch (this.options.appType) {
      case 'express':
        expressHandler.registerRoute.call(this)
        break;
      case 'koa':
        koaHandler.registerRoute.call(this)
        break;
      case 'egg':
        eggHandler.registerRoute.call(this)
        break;
      default:
        throw('NodeMonitorCheck Error: appType参数错误，可选参数express,koa,egg'); 
    }
  }
  onSignals() {

    // 想要处理的信号集合
    // 注意: SIGKILL信号（9）不能被截取和处理的
    const signals = {
      SIGHUP: 1,
      SIGINT: 2,
      SIGTERM: 15
    };

    // 在这里为我们的应用程序做必要的关闭逻辑
    const shutdown = (signal, value) => {
      this.stopServer((stdout) => {
        console.log(stdout)
      })
    };

    // 为我们想要处理的每个信号创建一个监听器
    Object.keys(signals).forEach(signal => {
      process.on(signal, () => {
        shutdown(signal, signals[signal]);
      });
    });
  }
  onConnection() {
    this.sockets = [];
    this.server.on('connection', (socket) => {
      this.sockets.push(socket);
      socket.once('close', () => {
        this.sockets.splice(this.sockets.indexOf(socket), 1);
      });
    });
  }
}

module.exports = NodeDocker;
