interface UseRequestOption<TRequest extends unknown[], TResult> {
  defaultParams?: TRequest
  mannual?: boolean
  onSuccess?: (data: TResult) => void
  onError?: (error: Error, params: TRequest) => void
  onFinally?: () => void
}

interface UseRequestResult<TRequest extends unknown[], TResult> {
  data?: Awaited<TResult>
  loading: boolean
  error?: Error
  run: (...args: TRequest) => Promise<TResult>
  refresh: () => Promise<TResult>
}

const useRequest = <TRequest extends unknown[], TResult>(
  request: (...args: TRequest) => TResult,
  options: UseRequestOption<TRequest, TResult> = {}
): UseRequestResult<TRequest, TResult> => {
  const [data, setData] = useState<Awaited<TResult>>()
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<Error>()

  const cacheParams = useRef<TRequest>(options.defaultParams || ([] as unknown as TRequest))

  const run = useCallback(async (...args: TRequest) => {
    setLoading(true)
    try {
      cacheParams.current = args
      const result = await request(...args)
      options.onSuccess?.(result)
      setData(result)
      return result
    } catch (err) {
      options.onError?.(err as Error, args)
      setError(err as Error)
      throw err
    } finally {
      options.onFinally?.()
      setLoading(false)
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [])

  const refresh = useCallback(() => {
    return run(...cacheParams.current)
  }, [run])

  useEffect(() => {
    if (options.mannual) {
      return
    }
    refresh()
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [])

  return {
    data,
    loading,
    error,
    run,
    refresh,
  }
}

export default useRequest
