import numeral from 'numeral'
import { numberToChineseUppercaseCurrency, currency } from 'pixiu-number-toolkit'

/**
 * 金额转换工具
 * /

/**
 * 
 * @param num 金额｜单位分
 * @description 1元 = 100分
 * @param precision 小数点位数
 * @returns 格式化的金额｜单位元
 * @description 默认保留2位小数
 */
const fenToYuan = (num: number | string, precision = 2): number | string => {
  if (isNothing(num)) {
    return ''
  } else {
    return numeral(num)
      .divide(100)
      .format(`0.${Array(precision).fill(0).join('')}`)
  }
}

/**
 *
 * @param num 金额｜单位分
 * @description 1元 = 100分
 * @param precision 小数点位数
 * @returns 格式化的金额 ｜单位元
 * @description 返回金额为整数，无小数点
 * @description 返回金额为小数，保留precision小数点
 */
const smartFenToYuan = (num: number | string, precision = 2): number | string => {
  if (isNothing(num)) {
    return ''
  }
  return numeral(num)
    .divide(100)
    .format(`0[.]${Array(precision).fill(0).join('')}`)
}

/**
 * @param num  金额｜单位厘
 * @description 1元 = 1000厘
 * @param precision 小数点位数
 * @returns 格式化的金额 ｜单位元
 * @description 默认保留2位小数
 */
const liToYuan = (num: number | string, precision = 2): number | string => {
  if (isNothing(num)) {
    return ''
  } else {
    return numeral(num)
      .divide(1000)
      .format(`0.${Array(precision).fill(0).join('')}`)
  }
}

/**
 *
 * @param num 金额｜单位元
 * @description 1元 = 100分
 * @param precision 小数点位数
 * @returns 格式化的金额｜单位分
 * 默认保留2位小数
 */
const yuanToFen = (num: number | string, precision = 2): number | string => {
  if (isNothing(num)) {
    return ''
  } else {
    return numeral(num)
      .multiply(100)
      .format(`0.${Array(precision).fill(0).join('')}`)
  }
}

/**
 * @param num  金额｜单位元
 * @description 1元 = 1000厘
 * @param precision 小数点位数
 * @returns 格式化的金额 ｜单位厘
 * @description 默认保留2位小数
 */
const yuanToLi = (num: number | string, precision = 2): number | string => {
  if (isNothing(num)) {
    return ''
  } else {
    return numeral(num)
      .multiply(1000)
      .format(`0.${Array(precision).fill(0).join('')}`)
  }
}

/**
 *
 * @param num 金额｜单位元
 * @returns 大写金额
 * @description 小数点最多读取2位，角、分
 * @description 负数，前面加"欠"
 */
const digitUppercase = (num: number | string): string => {
  if (isNothing(num)) {
    return ''
  } else {
    return numberToChineseUppercaseCurrency(Number(num))
  }
}

/**
 * 添加单位元
 * @param num 数字
 * @returns num + '元'
 * @description 默认保留2位小数
 */
const unitYuan = (num: string | number, precision = 2): string => {
  return currency(num, { suffix: '元', minimumFractionDigits: precision })
}

/**
 * 添加前缀¥
 * @param num 数字
 * @returns ¥num
 * @description 默认保留2位小数
 */
const prefixYuan = (num: string | number, precision = 2): string => {
  return currency(num, { prefix: '¥', minimumFractionDigits: precision })
}
export { fenToYuan, smartFenToYuan, liToYuan, yuanToFen, yuanToLi, digitUppercase, unitYuan, prefixYuan }
