import type { GetProp, UploadProps } from 'antd'

import { LoadingOutlined, PlusOutlined } from '@ant-design/icons'
import numeral from 'numeral'
type FileType = Parameters<GetProp<UploadProps, 'beforeUpload'>>[0]
const XCsrfToken = cookie.get('csrf_token') || ''

const defaultHeaders = {
  'X-Csrf-Token': XCsrfToken,
}

interface isLoading {
  loading: boolean
}
const UploadButton = (load: isLoading) => (
  <button style={{ border: 0, background: 'none' }} type='button'>
    {load.loading ? <LoadingOutlined /> : <PlusOutlined />}
    {!load.loading && <div className='ant-upload-text'>点击上传</div>}
  </button>
)

const ImgAccept = ['.jpg', '.jpeg', '.png']

const DefaultAccept = [...ImgAccept, '.mp4']

interface IDraggerUploadProps extends UploadProps {
  value?: string
  width?: number
  height?: number
  size?: number // 单位是mb
  videoSize?: number // 单位是mb
  videoDuration?: number // 单位秒s
  duration?: number
  displayZoomSize?: number // 默认预览图适配系数
  autoZoomSize?: boolean // 预览图是否自动适配图片宽高
  checkSize?: boolean //是否校验宽高
  render?: () => React.ReactNode // 自定义默认的上传背景内容
  getUploadUrl?: (url: string) => void // 上传成功后的参数回调
}

const DraggerUpload: React.FC<IDraggerUploadProps> = observer(
  (
    {
      value,
      width,
      height,
      checkSize = true,
      size = 5, // 默认是5mb
      videoSize,
      videoDuration,
      displayZoomSize = 1,
      autoZoomSize = false,
      getUploadUrl,

      name = 'file',
      action = 'https://ssp-manager.tuiatest.cn/upload/index',
      data,
      showUploadList = false,
      accept = DefaultAccept,

      beforeUpload = () => {
        return true
      },
      render,
      ...restProps
    },
    ref
  ) => {
    const [imgUrl, changeImgUrl] = useState(value)
    const [loading, changeLoading] = useState(false)
    const [messageApi, contextHolder] = message.useMessage()
    useEffect(() => {
      changeImgUrl(value)
    }, [value])

    const commonBeforeUpload: UploadProps['beforeUpload'] = async function (file: FileType, FileList: FileType[]) {
      return new Promise<void>(async (resolve, reject) => {
        // 默认校验
        if (checkIsImage(file)) {
          await getImgInfo(file).then(({ width: imgWidth, height: imgHeight, size: imgSize }) => {
            const isWidthValid = width ? imgWidth === width : true
            const isHeightValid = height ? imgHeight === height : true
            if ((!isWidthValid || !isHeightValid) && checkSize) {
              messageApi.error(height ? `请上传宽${width},高${height}的图片` : `请上传宽${width}的图片`)
              return false
            }
            if (imgSize > size * 1024 * 1024) {
              messageApi.error(`请上传不大于${numeral(size * 1024 * 1024).format('0b')}的图片`)
              return false
            }
            return true
          })
        }
        if (checkIsVideo(file)) {
          await getVideoInfo(file).then(({ duration = 0 }) => {
            if (videoSize && videoSize * 1024 * 1024 < file.size) {
              messageApi.error(`请上传不大于${numeral(videoSize * 1024 * 1024).format('0b')}的视频`)
              return false
            }
            if (videoDuration && videoDuration < duration) {
              messageApi.error(`请上传不长于${videoDuration}s的视频`)
              return false
            }
          })
        }
        // 自定义校验
        const canUpload = await beforeUpload?.(file, FileList)
        if (!canUpload) {
          reject(new Error('上传失败'))
          return
        }
        resolve()
        changeLoading(true)
        return true
      })
    }
    // 不手动修改上传内容（联调后看看需不需要在添加）
    const commonOnChange: UploadProps['onChange'] = (info) => {
      console.log('info', info)
      changeLoading(true)
      const {
        file: { status, response },
      } = info
      if (status === 'error') {
        changeLoading(false)
        changeImgUrl('http://yun.dui88.com/-rr2kxnz9cm-e1lpywd8t.png')
        getUploadUrl?.('http://yun.dui88.com/-rr2kxnz9cm-e1lpywd8t.png')
        messageApi.error(`${info.file.name} 上传失败，请重新上传。`)
      }

      // 接口
      // if (status !== 'uploading') {
      //   if (response) {
      //     const { data, code, msg } = response
      //     if (code === 200) {
      //       getUploadUrl?.(data.url)
      //       changeLoading(false)
      //     } else {
      //       messageApi.error(msg)
      //     }
      //   }
      // }
    }

    const displayStyle = useMemo(() => {
      if (!width || !height) return {}
      let zoomSize = displayZoomSize
      if (autoZoomSize) {
        if (width > 300) {
          zoomSize = 0.5
        }
        if (width > 500) {
          zoomSize = 0.3
        }
      }
      return { width: width * zoomSize, height: height * zoomSize }
    }, [width, height, autoZoomSize, displayZoomSize])

    return (
      <>
        {contextHolder}
        <Upload
          ref={ref}
          multiple={false}
          name={name}
          showUploadList={showUploadList}
          action={action}
          data={data}
          headers={defaultHeaders}
          accept={Array.isArray(accept) ? accept?.join(',') : accept}
          disabled={loading || restProps.disabled}
          {...restProps}
          beforeUpload={commonBeforeUpload}
          onChange={commonOnChange}>
          {render ? (
            imgUrl && !loading ? (
              <div style={{ width: 192, margin: 16 }}>
                <img src={value} alt='avatar' style={displayStyle} />
              </div>
            ) : (
              render()
            )
          ) : (
            <div style={displayStyle}>
              {imgUrl && !loading ? <img src={imgUrl} style={displayStyle} /> : <UploadButton loading={loading} />}
            </div>
          )}
        </Upload>
      </>
    )
  },
  {
    forwardRef: true,
  }
)

export default DraggerUpload
