/**
 * 常用正则表达式库
 */
import { REGEX, buildRegex } from 'pixiu-number-toolkit'

const isEmail = (email: string): boolean => {
  if (isNothing(email)) {
    return false
  } else {
    return REGEX.EMAIL.test(email)
  }
}

/**
 * @param REGEX 正则表达式= [
 * https://www.muchappy.com/open_source/pixiu-number-toolkit/basic/regex
 * NUMBER: 基础数字，包括正数、负数和小数
 * INTEGER: 整数，包括正整数、负整数和零
 * POSITIVE_INTEGER: 正整数
 * NEGATIVE_INTEGER: 负整数
 * NON_POSITIVE_INTEGER: 非正整数（包括负数和零）
 * FLOAT: 浮点数
 * POSITIVE_FLOAT: 正浮点数
 * NEGATIVE_FLOAT: 负浮点数
 * EMAIL: 电子邮箱地址
 * CHINESE_NAME: 中文姓名
 * ENGLISH_LETTER: 英文字母
 * LOWERCASE_ENGLISH_LETTER: 小写英文字母
 * UPPERCASE_ENGLISH_LETTER: 大写英文字母
 * ]
 * @param value 待验证的值
 * @description 通过正则表达式验证值是否符合规则
 * @returns true：符合规则 false：不符合规则
 */
const buildREG = (REGEX: RegExp, value: string | number): boolean => {
  const strictRegex = buildRegex(REGEX, { strict: true })
  return strictRegex.test(value + '')
}

const Rules = {
  required: {
    required: true,
    message: '必填',
  },
  id: {
    pattern: /^\d{0,20}$/,
    message: '只能是数字',
  },
  ids: {
    pattern: /^\d{0,20}(,\d{0,20})*$/,
    message: '输入的格式有误',
  },
  name: {
    max: 20,
    message: '最多不超过20个字',
  },
  noCN: {
    pattern: /^[^\u4e00-\u9fa5]+$/,
    message: '不能出现中文',
  },
  email: {
    pattern: REGEX.EMAIL,
    message: '请输入正确的邮箱格式',
  },
  url: {
    pattern: REGEX.URL,
    message: '请输入正确的网址',
  },
  positiveInt: {
    pattern: REGEX.POSITIVE_INTEGER,
    message: '请输入正整数',
  },
  nonSpecialChar: {
    pattern: REGEX.NON_SPECIAL_CHARACTERS,
    message: '请输入非特殊字符',
  },
  intLt1e100: {
    pattern: /^([1-9][0-9]{0,1}|100)$/,
    message: '请输入1-100整数',
  },
  intLt0e100: {
    pattern: /^100$|^(\d|[1-9]\d)$/,
    message: '请输入0-100整数',
  },
  float0e100: {
    pattern: /^(?:100(?:\.00?)?|\d{1,2}(?:\.\d{1,2})?)$/,
    message: '请输入0-100之间的数字，最多保留两位小数',
  },
  // 金额校验  包含正负 0  两位小数
  money: {
    pattern: /(^([+-]?)[1-9]([0-9]+)?(\.[0-9]{1,2})?$)|(^([+-]?)(0){1}$)|(^([+-]?)[0-9]\.[0-9]([0-9])?$)/,
    message: '请输入两位小数以内的正负金额数',
  },
  percent: {
    pattern: /^(\d|[1-9]\d|100)(\.\d{1,2})?$/,
    message: '请输入正确的百分比',
  },

}

export { isEmail, buildREG, Rules }
