

import Filter from '../Filter';
import { defaultVert } from '../defaultVerts';

const alphaFrag =
    'precision mediump float;' +
    'varying vec2 vTextureCoord;' +

    'uniform sampler2D uSampler;' +
    'uniform float uAlpha;' +

    'void main(void){' +
    'gl_FragColor = texture2D(uSampler, vTextureCoord) * uAlpha;' +
    '}'
/**
 * Simplest filter - applies alpha
 *
 * Use this instead of Container's alpha property to avoid visual layering of individual elements.
 * AlphaFilter applies alpha evenly across the entire display object and any opaque elements it contains.
 * If elements are not opaque, they will blend with each other anyway.
 *
 * Very handy if you want to use common features of all filters:
 *
 * 1. Assign a blendMode to this filter, blend all elements inside display object with background.
 *
 * 2. To use clipping in display coordinates, assign a filterArea to the same container that has this filter.
 *
 * @class
 * @extends Filter
 * @memberof filters
 */
export default class AlphaFilter extends Filter {
    /**
     * @param {number} [alpha=1] Amount of alpha from 0 to 1, where 0 is transparent
     */
    constructor(alpha: number = 1.0) {
        super(defaultVert, alphaFrag);
        this.alpha = alpha;
        this.glShaderKey = 'alpha';
    }

    /**
     * Coefficient for alpha multiplication
     *
     * @member {number}
     * @default 1
     */
    get alpha() {
        return this.uniforms.uAlpha;
    }
    set alpha(value: number) {
        this.uniforms.uAlpha = value;
    }
}
