import { calculatePlaneIndices } from './Plane';
import { Texture } from '../texture';
import CanvasRenderer from '../renderers/CanvasRenderer';
import { Mesh } from './Mesh';

//提前计算好的索引
const indicesNN = calculatePlaneIndices(4, 4)
/**
 * 九宫格
 *```js
 * let Plane9 = new NineSlicePlane(Texture.fromUrl('BoxWithRoundedCorners.png'), 15, 15, 15, 15);
 *  ```0,1,3,5,7,8,16,24,都是0
 * 
 * 0...1...2...3
 * .   .   .   .
 * 4...5...6...7
 * .   .   .   .
 * 8...9...10..11
 * .   .   .   .
 * 12..13..14..15
 * <pre>
 *      A                          B
 *    +---+----------------------+---+
 *  C | 1 |          2           | 3 |
 *    +---+----------------------+---+
 *    |   |                      |   |
 *    | 4 |          5           | 6 |
 *    |   |                      |   |
 *    +---+----------------------+---+
 *  D | 7 |          8           | 9 |
 *    +---+----------------------+---+
 * 当修改宽高width height时，不再改变scale属性
 * 1 3 7 9 永远不变
 * 2 8 只会水平横向拉伸
 * 4 6 只会垂直纵向拉伸
 * 5 都会拉伸
 * </pre>
 *
 *
 */
export class NineSlicePlane extends Mesh {
    /**
     * 宽高get set都重写，不再修改缩放，修改uv和顶点
     */
    get width(): number {
        return this._width;
    }
    set width(value: number) {
        this._width = value;
        this._needRefresh = true;
    }
    get height(): number {
        return this._height;
    }
    set height(value: number) {
        this._height = value;
        this._needRefresh = true;
    }
    /**
     * 原始宽度
     */
    private _origWidth: number;
    /**
     * 原始高度
     */
    private _origHeight: number;
    /**
     * 左边宽度
     */
    private _leftWidth: number;
    get leftWidth(): number {
        return this._leftWidth;
    }
    set leftWidth(value: number) {
        this._leftWidth = value;
        this._needRefresh = true;
    }
    /**
     * 右边宽度
     */
    private _rightWidth: number;
    get rightWidth(): number {
        return this._rightWidth;
    }
    set rightWidth(value: number) {
        this._rightWidth = value;
        this._needRefresh = true;
    }
    /**
     * 上边高度
     */
    private _topHeight: number;
    get topHeight(): number {
        return this._topHeight;
    }
    set topHeight(value: number) {
        this._topHeight = value;
        this._needRefresh = true;
    }
    /**
     * 下边高度
     */
    private _bottomHeight: number;
    get bottomHeight(): number {
        return this._bottomHeight;
    }
    set bottomHeight(value: number) {
        this._bottomHeight = value;
        this._needRefresh = true;
    }
    /**
     * @param {Texture} texture
     * @param {number} [leftWidth=10] 
     * @param {number} [topHeight=10] 
     * @param {number} [rightWidth=10] 
     * @param {number} [bottomHeight=10] 
     */
    constructor(texture: Texture, leftWidth: number = 10, topHeight: number = 10, rightWidth: number = 10, bottomHeight: number = 10) {
        super(texture);
        //考虑对于未加载好的图片怎么处理吧，肯定需要加在onTextureUpdate的
        this._origWidth = texture.orig.width;
        this._origHeight = texture.orig.height;

        this._width = this._origWidth;

        this._height = this._origHeight;

        this._leftWidth = leftWidth;

        this._rightWidth = rightWidth;

        this._topHeight = topHeight;

        this._bottomHeight = bottomHeight;

        //计算索引，完全不用变，所以提前计算
        this._indices = indicesNN;
        //顶点数量长度可以确定先
        this._vertices = new Float32Array(4 * 4 * 2);
        //uv长度也可以确定先
        this._uvs = new Float32Array(4 * 4 * 2);

        this.refresh(true);
    }

    /**
     * 额外增加修改原始宽高
     */
    protected _onTextureUpdate() {
        super._onTextureUpdate();
        this._origWidth = this.texture.orig.width;
        this._origHeight = this.texture.orig.height;
    }

    /**
     * 计算横向顶点
     */
    private updateHorizontalVertices() {
        const vertices = this._vertices;

        const h = this._topHeight + this._bottomHeight;
        const scale = this._height > h ? 1.0 : this._height / h;

        vertices[9] = vertices[11] = vertices[13] = vertices[15] = this._topHeight * scale;
        vertices[17] = vertices[19] = vertices[21] = vertices[23] = this._height - (this._bottomHeight * scale);
        vertices[25] = vertices[27] = vertices[29] = vertices[31] = this._height;
    }

    /**
     * 计算纵向顶点
     */
    private updateVerticalVertices() {
        const vertices = this._vertices;

        const w = this._leftWidth + this._rightWidth;
        const scale = this._width > w ? 1.0 : this._width / w;

        vertices[2] = vertices[10] = vertices[18] = vertices[26] = this._leftWidth * scale;
        vertices[4] = vertices[12] = vertices[20] = vertices[28] = this._width - (this._rightWidth * scale);
        vertices[6] = vertices[14] = vertices[22] = vertices[30] = this._width;
    }

    /**
     * 
     * 考虑是否用缓存，不然每次相当于9次绘制，到时应该是集成到一个插件里的
     * @private
     * @param {CanvasRenderer} renderer
     */
    _renderCanvas(renderer: CanvasRenderer) {
        const context = renderer.context;
        context.globalAlpha = this._worldAlpha;
        renderer.setBlendMode(this.blendMode);
        const transform = this.worldMatrix;
        context.setTransform(
            transform.a,
            transform.b,
            transform.c,
            transform.d,
            transform.tx,
            transform.ty
        );
        const base = this.texture.baseTexture;
        const textureSource = base.source;
        const w = base.width;
        const h = base.height;

        this.drawSegment(context, textureSource, w, h, 0, 1, 10, 11);
        this.drawSegment(context, textureSource, w, h, 2, 3, 12, 13);
        this.drawSegment(context, textureSource, w, h, 4, 5, 14, 15);
        this.drawSegment(context, textureSource, w, h, 8, 9, 18, 19);
        this.drawSegment(context, textureSource, w, h, 10, 11, 20, 21);
        this.drawSegment(context, textureSource, w, h, 12, 13, 22, 23);
        this.drawSegment(context, textureSource, w, h, 16, 17, 26, 27);
        this.drawSegment(context, textureSource, w, h, 18, 19, 28, 29);
        this.drawSegment(context, textureSource, w, h, 20, 21, 30, 31);
    }

    /**
     * 分部画
     * 每部分保证至少有1像素
     * @private
     * @param {CanvasRenderingContext2D} context 
     * @param {CanvasImageSource} textureSource - 图片资源
     * @param {number} w - 贴图宽
     * @param {number} h - 贴图高
     * @param {number} x1 - x index 1
     * @param {number} y1 - y index 1
     * @param {number} x2 - x index 2
     * @param {number} y2 - y index 2
     */
    private drawSegment(
        context: CanvasRenderingContext2D,
        textureSource: HTMLImageElement | HTMLCanvasElement,
        w: number,
        h: number,
        x1: number,
        y1: number,
        x2: number,
        y2: number
    ) {
        const uvs = this._uvs;
        const vertices = this._vertices;

        let sw = (uvs[x2] - uvs[x1]) * w;
        let sh = (uvs[y2] - uvs[y1]) * h;
        let dw = vertices[x2] - vertices[x1];
        let dh = vertices[y2] - vertices[y1];

        //绘制源保证至少有一像素
        if (sw < 1) sw = 1;
        if (sh < 1) sh = 1;

        //绘制体保证至少一像素
        if (dw < 1) dw = 1;
        if (dh < 1) dh = 1;

        context.drawImage(textureSource, uvs[x1] * w, uvs[y1] * h, sw, sh, vertices[x1], vertices[y1], dw, dh);
    }

    /**
     * 计算所有坐标
     */
    _refresh() {

        const uvs = this._uvs;
        const texture = this.texture;

        this._origWidth = texture.orig.width;
        this._origHeight = texture.orig.height;

        const _uvw = 1.0 / this._origWidth;
        const _uvh = 1.0 / this._origHeight;

        uvs[0] = uvs[8] = uvs[16] = uvs[24] = 0;
        uvs[1] = uvs[3] = uvs[5] = uvs[7] = 0;
        uvs[6] = uvs[14] = uvs[22] = uvs[30] = 1;
        uvs[25] = uvs[27] = uvs[29] = uvs[31] = 1;

        uvs[2] = uvs[10] = uvs[18] = uvs[26] = _uvw * this._leftWidth;
        uvs[4] = uvs[12] = uvs[20] = uvs[28] = 1 - (_uvw * this._rightWidth);
        uvs[9] = uvs[11] = uvs[13] = uvs[15] = _uvh * this._topHeight;
        uvs[17] = uvs[19] = uvs[21] = uvs[23] = 1 - (_uvh * this._bottomHeight);

        this.updateHorizontalVertices();
        this.updateVerticalVertices();

        this._vertexDirty++;

        this.multiplyUvs();
    }
}
