import { hex2string, hex2rgb } from '../utils';
import { Matrix, Rectangle } from '../math';
import { RENDERER_TYPE, devicePixelRatio } from '../const';
import Container from '../display/Container';
import { EventDispatcher } from '../events/EventDispatcher';
import { DisplayObject } from "../display/DisplayObject";
import { RendererOptions } from "./RendererOptions";

const tempMatrix = new Matrix();

/**
 * The SystemRenderer is the base for a  Renderer. It is extended by the {@link CanvasRenderer}
 * and {@link WebGLRenderer} which can be used for rendering a  scene.
 * @abstract
 * @class
 */
export default class SystemRenderer extends EventDispatcher {
    /**
     * 渲染器参数
     */
    options: RendererOptions;
    /**
     * 渲染类型
     *
     * @member {number}
     * @default RENDERER_TYPE.UNKNOWN
     * @see RENDERER_TYPE
     */
    type: number;
    /**
     * canvas对象
     */
    htmlElement: HTMLCanvasElement;
    /**
     * 是否透明
     */
    transparent: boolean;
    /**
     * The value of the preserveDrawingBuffer flag affects whether or not the contents of
     * the stencil buffer is retained after rendering.
     */
    preserveDrawingBuffer: boolean;
    /**
     * 背景色，十六进制数值
     * @member {number}
     */
    protected _backgroundColor: number;
    /**
     * 背景色，rgba值
     * @member {number[]}
     */
    protected _backgroundColorRgba: number[];
    /**
     * 背景色，字符串
     */
    protected _backgroundColorString: string;
    /**
     * 临时父级对象，用于更新updateTransform
     */
    protected _tempDisplayObjectParent: Container;
    /**
     * 上一个被渲染的根显示对象
     */
    _lastObjectRendered: DisplayObject;

    constructor(options: RendererOptions) {
        super();

        this.options = options;

        this.type = RENDERER_TYPE.UNKNOWN;

        this.htmlElement = options.htmlElement;

        this.transparent = options.transparent;

        this.preserveDrawingBuffer = options.preserveDrawingBuffer;

        this._backgroundColor = 0x000000;

        this._backgroundColorRgba = [0, 0, 0, 0];

        this._backgroundColorString = '#000000';

        this.backgroundColor = options.backgroundColor || this._backgroundColor; // run bg color setter

        this._tempDisplayObjectParent = new Container();

        this._lastObjectRendered = this._tempDisplayObjectParent;
    }

    /**
     * 尺寸重置，暂时根据屏幕分辨率
     * @param {number} screenWidth
     * @param {number} screenHeight
     */
    resize(width: number, height: number) {
        this.htmlElement.width = width * devicePixelRatio;
        this.htmlElement.height = height * devicePixelRatio;
        this.htmlElement.style.width = `${width}px`;
        this.htmlElement.style.height = `${height}px`;
    }

    /**
     * 核心渲染方法，子级重写
     */
    render(displayObject, renderTexture?, transform?) {

    }

    /**
     * 销毁方法
     */
    destroy() {
        this.type = RENDERER_TYPE.UNKNOWN;

        this.htmlElement = null;

        this.transparent = false;
        this.options = null;
        this.preserveDrawingBuffer = false;
        this._backgroundColor = 0;
        this._backgroundColorRgba = null;
        this._backgroundColorString = null;

        this._tempDisplayObjectParent = null;
        this._lastObjectRendered = null;
    }

    /**
     * 背景色，十六进制颜色
     * @member {number}
     */
    get backgroundColor(): number {
        return this._backgroundColor;
    }

    /**
     * 背景色，十六进制颜色
     */
    set backgroundColor(value: number) {
        if (this._backgroundColor === value) return
        this._backgroundColor = value;
        this._backgroundColorString = hex2string(value);
        hex2rgb(value, this._backgroundColorRgba);
    }
}
