/**
 * Created by rockyl on 2018/11/7.
 *
 * 交互上下文
 */

let _canvas;
let _touchHandler;
let _scaleX, _scaleY, _rotation;

const ua = navigator.userAgent.toLowerCase();
const isMobile = (ua.indexOf('mobile') !== -1 || ua.indexOf('android') !== -1);

/**
 * 装配上下文
 * @param options
 */
export function setupContext(options:any = {}) {
	const {canvas, touchHandler} = options;

	_touchHandler = touchHandler;

	_canvas = canvas;

	addListeners();
}

/**
 * 更新缩放模式
 * @param scaleX
 * @param scaleY
 * @param rotation
 */
export function updateScaleMode(scaleX, scaleY, rotation) {
	_scaleX = scaleX;
	_scaleY = scaleY;
	_rotation = rotation;
}

/**
 * 适配鼠标事件
 */
function addListeners() {
	if (window.navigator.msPointerEnabled) {
		_canvas.addEventListener("MSPointerDown", (event) => {
			event.identifier = event.pointerId;
			onTouchBegin(event);
			prevent(event);
		}, false);
		_canvas.addEventListener("MSPointerMove", (event) => {
			event.identifier = event.pointerId;
			onTouchMove(event);
			prevent(event);
		}, false);
		_canvas.addEventListener("MSPointerUp", (event) => {
			event.identifier = event.pointerId;
			onTouchEnd(event);
			prevent(event);
		}, false);
	}
	else {
		if (!isMobile) {
			addMouseListener();
		}
		addTouchListener();
	}
}

/**
 * 阻断页面拖动
 * @param event
 */
function prevent(event) {
	event.stopPropagation();
	if (event["isScroll"] != true && !_canvas['userTyping']) {
		event.preventDefault();
	}
}

/**
 * 增加鼠标事件
 */
function addMouseListener() {
	_canvas.addEventListener("mousedown", onTouchBegin);
	_canvas.addEventListener("mousemove", onMouseMove);
	_canvas.addEventListener("mouseup", onTouchEnd);
}

/**
 * 增加触摸事件
 */
function addTouchListener() {
	_canvas.addEventListener("touchstart", (event) => {
		let l = event.changedTouches.length;
		for (let i = 0; i < l; i++) {
			onTouchBegin(event.changedTouches[i]);
		}
		prevent(event);
	}, false);
	_canvas.addEventListener("touchmove", (event) => {
		let l = event.changedTouches.length;
		for (let i = 0; i < l; i++) {
			onTouchMove(event.changedTouches[i]);
		}
		prevent(event);
	}, false);
	_canvas.addEventListener("touchend", (event) => {
		let l = event.changedTouches.length;
		for (let i = 0; i < l; i++) {
			onTouchEnd(event.changedTouches[i]);
		}
		prevent(event);
	}, false);
	_canvas.addEventListener("touchcancel", (event) => {
		let l = event.changedTouches.length;
		for (let i = 0; i < l; i++) {
			onTouchEnd(event.changedTouches[i]);
		}
		prevent(event);
	}, false);
}

function onTouchBegin(event) {
	let location = getLocation(event);
	_touchHandler.onTouchBegin(location);
}

function onMouseMove(event) {
	if (event.buttons === 0) {
		onTouchEnd(event);
	} else {
		onTouchMove(event);
	}
}

function onTouchMove(event) {
	let location = getLocation(event);
	_touchHandler.onTouchMove(location);

}

function onTouchEnd(event) {
	let location = getLocation(event);
	_touchHandler.onTouchEnd(location);
}

/**
 * 获取当前点
 */
function getLocation(event) {
	let doc = document.documentElement;
	let box = _canvas.getBoundingClientRect();
	let left = box.left + window.pageXOffset - doc.clientLeft;
	let top = box.top + window.pageYOffset - doc.clientTop;
	let x = event.pageX - left, newX = x;
	let y = event.pageY - top, newY = y;
	if (_rotation === 90) {
		newX = y;
		newY = box.width - x;
	}
	else if (_rotation === -90) {
		newX = box.height - y;
		newY = x;
	}
	newX = newX / _scaleX;
	newY = newY / _scaleY;
	return {
		x: Math.round(newX),
		y: Math.round(newY),
		identifier: event.identifier || 0,
	};
}