// Learn TypeScript:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/typescript.html
// Learn Attribute:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/reference/attributes.html
// Learn life-cycle callbacks:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/life-cycle-callbacks.html

import { LevelInfo } from "./Config/GameConfig";
import { numToChinese, set16ToRgb } from "./utils";
const { ccclass, property } = cc._decorator;

// 爪子状态
const CLIP_STATE = {
  /** 默认 */
  DEFAULT: 'default',
  /** 出钩 */
  PLAY: 'clipPlay',
  /** 收回 */
  STOP: 'clipStop'
}

@ccclass
export default class NewClass extends cc.Component {
  /** 单格进度条 */
  @property(cc.Prefab)
  procItem: cc.Prefab = null

  /** 当前关卡 0、1、2... */
  actLevel = 0,

  /** 关卡目标数量 */
  levelObjectives = 0

  /** 当前关卡信息 */
  actLevelInfo = null

  /** 已获得星星数量 */
  starNum = 0

  /** 倒计时 */
  countDown = 0

  /** 游戏是否结束 */
  isGameOver = false

  /** 爪子旋转动画 */
  clipAni: cc.Animation = null

  clipState = CLIP_STATE.STOP

  start() {
    this.refreshLevelInfo()
    this.clipAni = cc.find('clip/clipWrp/defa', this.node).getComponent(cc.Animation)
    this.addNodeEvent()
    this.node.on("test", this.onAnimCompleted,this)
  }

  /** 爪子事件 */
  onAnimCompleted({ detail }) {
    console.log(detail.state)
    if (detail.state === CLIP_STATE.PLAY) {
      this.setClipState(CLIP_STATE.STOP)
      return
    }
    if (detail.state === CLIP_STATE.STOP) {
      this.setClipState(CLIP_STATE.DEFAULT)
      return
    }
  }

  /** 添加节点的事件 */
  addNodeEvent() {
    const gameBtn = cc.find('gameBtn', this.node)
    gameBtn.on(cc.Node.EventType.TOUCH_END, this.playGame, this)
    const clip = this.clipAni
    clip.on('palyEnd', () => {
      console.log('end')
    }, this)
    clip.on('stop', () => {

    }, this)
  }

  /** 设置爪子状态 */
  setClipState(state) {
    const ani = cc.find('clip/clipWrp', this.node).getComponent(cc.Animation)
    const node = cc.find('clip/clipWrp/defa', this.node)
    const clip = node.getComponent(cc.Animation), spf = node.getComponent(cc.Sprite).spriteFrame
    spf.insetBottom = 125
    spf.insetTop = 10
    this.clipState = state
    switch (state) {
      case CLIP_STATE.DEFAULT:
        ani.resume()
        break;
      case CLIP_STATE.PLAY:
        ani.pause()
        clip.play(state)
        break;
      case CLIP_STATE.STOP:
        ani.pause()
        clip.play(state)
        break;
    }
  }

  /** 出钩 */
  playGame() {
    if (this.isGameOver) return
    this.setClipState(CLIP_STATE.PLAY)

    this.setStarProc()
    if (this.starNum >= this.levelObjectives) {
      const nextLevel = this.actLevel + 1
      if (nextLevel >= LevelInfo.length) { // 已通关
        this.gameOver()
        return
      }

      console.log(`过关`)
      this.refreshLevelInfo(nextLevel)
      return
    }
  }

  /** 游戏结束 */
  gameOver() {
    this.isGameOver = true
    this.unschedule(this.startCd)
  }

  /**
   * 设置节点lable
   * @param value 
   */
  setLable(key, value) {
    cc.find(key, this.node).getComponent(cc.Label).string = value + ''
  }

  /** 开始倒计时 */
  startCd() {
    const cd = this.countDown - 1 || 0
    if (!cd || cd <= 0) {
      this.setLable('cdIcon/cd', `0s`)
      console.log('倒计时结束')
      this.unschedule(this.startCd)
      return
    }
    this.countDown = cd
    this.setLable('cdIcon/cd', `${this.countDown}s`)
  }

  protected onDestroy(): void {
    this.unschedule(this.startCd)
  }

  /** 刷新关卡信息 */
  refreshLevelInfo(level = 0) {
    this.isGameOver = false
    this.actLevel = level
    this.starNum = 0
    this.levelObjectives = LevelInfo[level].colors.length
    this.actLevelInfo = LevelInfo[level]
    this.countDown = this.actLevelInfo.countDown

    this.setLable('levelName', `第${numToChinese(level + 1)}关`)
    this.setLable('starIcon/starProc', `${this.starNum}/${this.levelObjectives}`)

    this.updateGameProc()

    this.unschedule(this.startCd)
    this.setLable('cdIcon/cd', `${this.countDown}s`)
    this.schedule(this.startCd, 1)
  }

  /** 设置星星进度 */
  setStarProc() {
    const proc = cc.find('procBg/proc', this.node)
    const key = this.starNum
    if (key >= this.levelObjectives) return

    const node = cc.instantiate(this.procItem)
    const rgbAry = set16ToRgb(this.actLevelInfo.colors[key])
    node.color = new cc.Color(...rgbAry)
    node.setParent(proc)
    this.starNum++

    this.setLable('starIcon/starProc', `${this.starNum}/${this.levelObjectives}`)
  }

  /** 更新游戏进度条信息 */
  updateGameProc() {
    const procBg = this.node.getChildByName('procBg'), proc = procBg.getChildByName('proc')
    const actIdx = this.levelObjectives, procItemWidth = 39, procOffset = 1
    proc.removeAllChildren()
    procBg.width = procItemWidth * actIdx + 12 + (actIdx - 1) * procOffset
    proc.width = procItemWidth * actIdx + (actIdx - 1) * procOffset
    procBg.x = -(procBg.width / 2)
  }

  update(dt) {
  }
}
