// Learn TypeScript:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/typescript.html
// Learn Attribute:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/reference/attributes.html
// Learn life-cycle callbacks:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/life-cycle-callbacks.html

import { BLOCK_STATE, CUSTOM_EVENT, Config } from "./Config/GameConfig";
import propPool from "./propPool";
import { getRandomArrayElements, waitTime } from "./utils";

const { ccclass, property } = cc._decorator;

@ccclass
export default class BlockManager extends cc.Component {
  isMove = false

  viewWidth = 0

  viewHeight = 0

  /** 方块矩阵外层容器 */
  blockMatrix: cc.Node = null

  /** 默认位置 */
  defaultPos: cc.Vec2 = null

  /** 加分事件 */
  addScoreEvents: cc.Event.EventCustom = null

  /** 小方块 */
  @property(cc.Prefab)
  miniBlockPre: cc.Prefab = null

  start() {
    this.viewHeight = cc.view.getVisibleSize().height
    this.viewWidth = cc.view.getVisibleSize().width
    this.defaultPos = this.node.getPosition()
    this.addScoreEvents = new cc.Event.EventCustom(CUSTOM_EVENT.ADD_SCORE, true)
    this.blockMatrix = cc.find('blockMatrix', this.node.parent)
    this.refreshMiniBlock()
    this.addNodeEvent()
  }

  /** 刷新小方块的生成 */
  refreshMiniBlock() {
    const miniBlockMatrix = getRandomArrayElements(Config.miniBlockMatrix, 1)[0]
    this.node.removeAllChildren()
    const { miniBlockOffset, miniBlockW } = Config
    const len = miniBlockMatrix.length, maxRowNum = 0
    const maxColNumAry = Array(len).fill(0)
    miniBlockMatrix.forEach((colBlocks, i) => {
      let maxX = 0
      colBlocks.map((miniBlock, j) => {
        if (miniBlock == 1) {
          maxColNumAry[j]++
          maxX++
          const node = propPool.isEmpyt ? cc.instantiate(this.miniBlockPre) : propPool.pop()
          const miniBlockSize = miniBlockOffset + miniBlockW
          node.x = j * miniBlockSize - this.node.width / 2 + miniBlockOffset + miniBlockW / 2
          node.y = (len - i - 1) * miniBlockSize - this.node.width / 2 + miniBlockOffset + miniBlockW / 2
          node.setParent(this.node)
        }
      })

      if (maxX > maxRowNum) {
        maxRowNum = maxX
      }
    })
    // 处理偶数方块x轴偏移
    if (maxRowNum % 2 === 0) {
      this.node.children.forEach((n) => {
        n.x -= miniBlockW / 2
      })
    }
    // 处理偶数方块y轴偏移
    if (Math.max(...maxColNumAry) % 2 === 0) {
      this.node.children.forEach((n) => {
        n.y += miniBlockW / 2
      })
    }
  }

  /** 设置所有的方块复原 */
  setBlockMatrixAll() {
    this.isMove = false
    const allBlock = this.node.children
    if (this.isSetBlockMartix()) {
      allBlock.forEach((node: cc.Node) => {
        const scripts = node.getComponent('miniBlock')
        scripts.setBlockMatrix()
      })
      this.checkBlockMatrix()
      this.clearAllChildren()
      // this.refreshMiniBlock()
    }
    this.node.setPosition(this.defaultPos)
    this.node.scale = 1
  }


  /** 清除所有自己的子元素 */
  clearAllChildren() {
    this.node.children.forEach((node) => {
      propPool.push(node)
    })
    this.node.removeAllChildren()
  }

  /** 检查块矩阵是否能够消除 */
  checkBlockMatrix() {
    const blockMatrix = this.blockMatrix.children.map((node) => {
      const script = node.getComponent('block')
      return {
        node,
        script,
        blockState: script.blockState,
        idx: node.getSiblingIndex()
      }
    })
    const { maxCol, maxRow } = Config
    const eliminateBlocks = []
    // 检查行
    for (let i = 0; i < maxCol; i++) {
      let temp = []
      for (let j = 0; j < maxRow; j++) {
        const block = blockMatrix[j + i * maxCol]
        if (block.blockState !== BLOCK_STATE.NON_EMPTY) break
        temp.push(block)
        if (temp.length === maxRow) {
          // console.log('最后一个方块',block.idx)
          eliminateBlocks.push(temp)
        }
      }
    }

    // 检查列
    for (let i = 0; i < maxRow; i++) {
      let temp = []
      for (let j = 0; j < maxCol; j++) {
        const block = blockMatrix[i + j * maxRow]
        if (block.blockState !== BLOCK_STATE.NON_EMPTY) break
        temp.push(block)
        if (temp.length === maxCol) {
          // console.log('最后一个方块',block.idx)
          eliminateBlocks.push(temp)
        }
      }
    }

    this.clearMatrixBlock(eliminateBlocks)
    // console.log('eliminateBlocks:', eliminateBlocks)
  }

  /**
   * 清理矩阵块
   * @param eliminateBlocks 需要清理的方块组
   */
  clearMatrixBlock(eliminateBlocks: cc.Node[]) {
    const recursionPlay = async (els) => {
      if (els.length === 0) return
      const { script } = els.pop()
      script.playClearAni(() => {
        cc.find('Canvas').dispatchEvent(this.addScoreEvents)
      })
      await waitTime(Config.blockAniDelay)
      recursionPlay(els)
    }
    eliminateBlocks.forEach((els) => {
      recursionPlay(els)
    })
  }

  /**
   * 判断是否能够放入
   * @returns 
   */
  isSetBlockMartix() {
    const allBlock = this.node.children
    const filterBlock = []
    for (let i = 0; i < allBlock.length; i++) {
      const script = allBlock[i].getComponent('miniBlock')
      const { targetPos, targetNode } = script.getTargetMatrix()
      // 未获取到指定最短距离内的节点
      if (!targetNode) {
        return false
      }
      filterBlock.push({ targetPos, targetNode })
    }
    // 该目标节点是否为空
    const isEmpty = filterBlock.filter(({ targetNode }) => {
      const script = targetNode.getComponent('block')
      return script.blockState === BLOCK_STATE.NON_EMPTY
    }).length === 0
    if (!isEmpty) return false
    const allTargetNodeId = filterBlock.map(({ targetNode }) => targetNode.getSiblingIndex())
    // 索引是否重复
    const isRepeat = allTargetNodeId.find((_, i) => allTargetNodeId.includes(_, i + 1))
    return !isRepeat && isEmpty
  }

  addNodeEvent() {
    this.node.on(cc.Node.EventType.TOUCH_START, () => {
      this.isMove = true
      this.node.scale = 1.7
    }, this)

    this.node.on(cc.Node.EventType.TOUCH_END, () => {
      this.setBlockMatrixAll()
    }, this)

    this.node.on(cc.Node.EventType.TOUCH_MOVE, (e: cc.Event.EventTouch) => {

      if (this.isMove) {
        const pos: cc.Vec2 = e.getPreviousLocation()
        const viewW = this.viewWidth, viewH = this.viewHeight
        this.node.setPosition(pos.x - (viewW / 2), pos.y - (812 - (1624 - viewH) / 2))
      }
    }, this)
  }

  // update (dt) {}
}
