//AudioMgr.ts
import { Node, AudioSource, AudioClip, resources, director, assetManager } from 'cc';

/**
 * @en
 * this is a sington class for audio play, can be easily called from anywhere in you project.
 * @zh
 * 这是一个用于播放音频的单件类，可以很方便地在项目的任何地方调用。
 */
export class AudioMgr {

    private static _ins: AudioMgr;
    public static get ins(): AudioMgr {
        if (!this._ins) {
            this._ins = new AudioMgr();
        }
        return this._ins;
    }

    private _audioSource: AudioSource;
    private _musicVolume = 1.0;
    private _musicVolumeScale = 1.0;
    private _soundVolume = 1.0;

    constructor() {
        //@en create a node as audioMgr
        //@zh 创建一个节点作为 audioMgr
        let audioMgr = new Node();
        audioMgr.name = '__audioMgr__';

        //@en add to the scene.
        //@zh 添加节点到场景
        director.getScene().addChild(audioMgr);

        //@en make it as a persistent node, so it won't be destroied when scene change.
        //@zh 标记为常驻节点，这样场景切换的时候就不会被销毁了
        director.addPersistRootNode(audioMgr);

        //@en add AudioSource componrnt to play audios.
        //@zh 添加 AudioSource 组件，用于播放音频。
        this._audioSource = audioMgr.addComponent(AudioSource);
    }

    public get audioSource() {
        return this._audioSource;
    }

    public set musicVolume(v: number) {
        this._musicVolume = v;
        this._audioSource.volume = this._musicVolume * this._musicVolumeScale;
    }

    public set soundVolume(v: number) {
        this._soundVolume = v;
    }

    /**
     * @en
     * play short audio, such as strikes,explosions
     * @zh
     * 播放短音频,比如 打击音效，爆炸音效等
     * @param sound clip or url for the audio
     * @param volume
     */
    playOneShot(sound: AudioClip | string, volume: number = 1, bundleName: string = 'resources') {
        if (sound instanceof AudioClip) {
            this._audioSource.volume = 1.0;
            this._audioSource.playOneShot(sound, volume * this._soundVolume);
        } else {
            let bundle = assetManager.getBundle(bundleName);
            bundle.load(sound, (err, clip: AudioClip) => {
                if (err) {
                    console.log(err);
                } else {
                    this._audioSource.volume = 1.0;
                    this._audioSource.playOneShot(clip, volume * this._soundVolume);
                }
            });
        }
    }

    /**
     * @en
     * play long audio, such as the bg music
     * @zh
     * 播放长音频，比如 背景音乐
     * @param sound clip or url for the sound
     * @param loop
     * @param volume
     * @param bundleName
     */
    play(sound: AudioClip | string, loop: boolean = false, volume: number = 1, bundleName: string = 'resources') {
        this._musicVolumeScale = volume;
        if (sound instanceof AudioClip) {
            this._audioSource.clip = sound;
            this._audioSource.loop = loop;
            this._audioSource.play();
            this.audioSource.volume = this._musicVolume * this._musicVolumeScale;
        } else {
            let bundle = assetManager.getBundle(bundleName);
            bundle.load(sound, (err, clip: AudioClip) => {
                if (err) {
                    console.log(err);
                } else {
                    this._audioSource.clip = clip;
                    this._audioSource.loop = loop;
                    this._audioSource.play();
                    this.audioSource.volume = this._musicVolume * this._musicVolumeScale;
                }
            });
        }
    }

    /**
     * stop the audio play
     */
    stop() {
        this._audioSource.stop();
    }

    /**
     * pause the audio play
     */
    pause() {
        this._audioSource.pause();
    }

    /**
     * resume the audio play
     */
    resume() {
        this._audioSource.play();
    }
}