import { LevelBase } from "@/pages/GamePage/Components/LevelBase.ts";
import { Assets, Sprite } from "pixi.js";
import { Ease, Tween } from "@/core/tween";
import { GameEvent, globalEvent } from "@/pages/GamePage/GameEvent.ts";

export class Level11 extends LevelBase {

  level: number = 11;

  A: Sprite;

  right: Sprite;
  error: Sprite;

  // 双指缩放相关属性
  private initialDistance: number = 0;
  private initialScale: number = 1;
  private isTouching: boolean = false;
  private touchPoints: Map<number, { x: number, y: number }> = new Map();

  onLoad() {
    super.onLoad();

    this.addChild(new Sprite(Assets.get(`level${this.level}/长颈鹿.png`)))
      .position.set(91, 765);

    this.A = this.addChild(new Sprite(Assets.get(`level${this.level}/冰箱.png`)));
    this.A.anchor.set(0.5);
    this.A.position.set(541, 956);
    this.A.scale.set(0.5746031746031746);

    this.right = this.addChild(new Sprite(Assets.get(`level${this.level}/right.png`)));
    this.right.position.set(642, 1141);
    this.right.alpha = 0;
    this.right.interactive = false;
    this.right.eventMode = "none";

    this.A.on("pointerdown", this.onPointerDown, this);
    this.A.on("globalpointermove", this.onPointerMove, this);
    this.A.on("pointerup", this.onPointerUp, this);
    this.A.on("pointerupoutside", this.onPointerUp, this);
  }

  onPointerDown(e) {
    console.log("pointerdown", e.pointerId, e.data.global);

    // 记录触摸点
    this.touchPoints.set(e.pointerId, {
      x: e.data.global.x,
      y: e.data.global.y
    });

    // 如果有两个触摸点，开始缩放
    if (this.touchPoints.size === 2) {
      const points = Array.from(this.touchPoints.values());
      this.initialDistance = this.getDistance(points[0], points[1]);
      this.initialScale = this.A.scale.x;
      this.isTouching = true;
      console.log("开始缩放", this.initialDistance, this.initialScale);
    }
  }

  onPointerMove(e) {
    if (!this.touchPoints.has(e.pointerId)) return;

    // 更新触摸点位置
    this.touchPoints.set(e.pointerId, {
      x: e.data.global.x,
      y: e.data.global.y
    });

    if (this.isTouching && this.touchPoints.size === 2) {
      const points = Array.from(this.touchPoints.values());
      const currentDistance = this.getDistance(points[0], points[1]);

      // 计算缩放比例
      const scaleRatio = currentDistance / this.initialDistance;
      const newScale = this.initialScale * scaleRatio;

      // 限制最小和最大缩放
      const finalScale = Math.max(0.5, Math.min(newScale, 5));
      this.A.scale.set(finalScale);

      console.log("缩放中", currentDistance, scaleRatio, finalScale);

      // 检查是否达到目标大小
      this.checkSize();
    }
  }

  onPointerUp(e) {
    console.log("pointerup", e.pointerId);

    // 移除触摸点
    this.touchPoints.delete(e.pointerId);

    // 如果少于两个触摸点，停止缩放
    if (this.touchPoints.size < 2) {
      this.isTouching = false;
      console.log("停止缩放");
    }
  }

  getDistance(point1: { x: number, y: number }, point2: { x: number, y: number }): number {
    const dx = point1.x - point2.x;
    const dy = point1.y - point2.y;
    return Math.sqrt(dx * dx + dy * dy);
  }

  checkSize() {
    if (this.A.scale.x >= 1) {
      this.triggerSuccess();
    }
  }

  triggerSuccess() {
    // 防止重复触发
    if (this.right.alpha > 0) return;

    this.setTouchEnable(false);

    Tween.get(this.right)
      .to({ alpha: 1 }, 444, Ease.quadInOut)
      .wait(2000)
      .call(() => {
        globalEvent.emit(GameEvent.NextLevel);
      });
  }

  onDestroy() {
    super.onDestroy();
    Tween.removeTweens(this.A);
    Tween.removeTweens(this.right);

    // 移除触摸事件监听
    this.A.off("pointerdown", this.onPointerDown, this);
    this.A.off("globalpointermove", this.onPointerMove, this);
    this.A.off("pointerup", this.onPointerUp, this);
    this.A.off("pointerupoutside", this.onPointerUp, this);
  }
}
