import {
    _decorator,
    math,
    v3,
    Vec3,
    Node,
    Collider2D,
    Contact2DType,
    PhysicsGroup,
    Vec2,
    IVec2Like,
    v2,
} from "cc";
import { Snake } from "./Snake";
import { DirectionType } from "./Common/Enums";
import { Global } from "./Global";
import { MainGame } from "./MainGame";
import { FoodManger } from "./Manager/FoodManger";

const { ccclass, property } = _decorator;

enum AIState {
    HUNTING,    // 追逐食物
    INTERCEPTING, // 拦截玩家
    ESCAPING,   // 逃离危险
    WANDERING,   // 随机游走
    ASSISTING   // 协助其他AI攻击玩家
}

@ccclass("AISnake")
export class AISnake extends Snake {
    @property({
        range: [1, 5],
        tooltip: "AI难度(1-5)"
    })
    private difficulty: number = 5;
    // private difficulty: number = 1 + Math.random() * 4;

    // AI状态相关
    private currentState: AIState = AIState.WANDERING;
    private targetFood: Vec2 = null;
    private targetSnake: Snake = null;
    private escapeTarget: Snake = null;
    private assistTarget: AISnake = null;


    // 计时器
    private behaviorTimer: number = 0;
    private updateTimer = 0;
    private currentUpdateInterval = 0.1;

    // 常量配置
    private readonly BASE_VIEW_DISTANCE = 300;  // 基础视野距离
    private readonly INTERCEPT_DISTANCE = 350;  // 拦截距离
    private readonly PREDICTION_TIME = 1.2;  // 预测时间
    private readonly ESCAPE_BOUNDARY = 250;  // 逃跑边界    
    private readonly SAFE_MARGIN = 3.0;  // 安全边际

    private readonly COLLISION_CHECK_DISTANCE = 500;  // 碰撞检测距离
    private readonly COLLISION_CHECK_DISTANCE_SQ = this.COLLISION_CHECK_DISTANCE * this.COLLISION_CHECK_DISTANCE;

    private readonly ASSIST_DISTANCE = 500;
    private readonly ASSIST_DISTANCE_SQ = this.ASSIST_DISTANCE * this.ASSIST_DISTANCE;

    private readonly DANGER_ANGLE_THRESHOLD = 90;
    private readonly COUNTER_ATTACK_THRESHOLD = 0.8;
    private readonly SAFE_DISTANCE_MULTIPLIER = 1.5;
    private readonly BASE_UPDATE_INTERVAL = 0.1;


    // 难度参数
    difficultyParams = {
        reactionTime: 0,
        viewDis: 0,
        viewDisSq: 0,
        interceptDis: 0,
        interceptDisSq: 0,
        aggressiveness: 0,
        predictionAccuracy: 0,
        turnSpeed: 0,
    };

    onLoad() {
        super.onLoad();
        const {
            difficulty,
            difficultyParams,
            BASE_VIEW_DISTANCE,
            INTERCEPT_DISTANCE,
        } = this;

        difficultyParams.reactionTime = math.lerp(0.8, 0.15, (difficulty - 1) / 4);

        difficultyParams.viewDis = BASE_VIEW_DISTANCE * (1 + (difficulty - 1) * 0.2);
        difficultyParams.viewDisSq = difficultyParams.viewDis * difficultyParams.viewDis;

        difficultyParams.interceptDis = INTERCEPT_DISTANCE * (1 + (difficulty - 1) * 0.2);
        difficultyParams.interceptDisSq = difficultyParams.interceptDis * difficultyParams.interceptDis;

        difficultyParams.aggressiveness = math.lerp(0.2, 0.7, (difficulty - 1) / 4);
        difficultyParams.predictionAccuracy = math.lerp(0.6, 0.9, (difficulty - 1) / 4);
        difficultyParams.turnSpeed = math.lerp(2, 4.5, (difficulty - 1) / 4);
    }

    onEnable() {
        super.onEnable();

        const eye = this.head.getChildByName("范围").getComponent(Collider2D);
        eye.on(Contact2DType.BEGIN_CONTACT, this.onBeginEye, this);

    }

    onBeginEye(selfCollider: Collider2D, otherCollider: Collider2D) {
        if (otherCollider.group === PhysicsGroup["Body"] && otherCollider.tag != this.tag) {
            // 碰到其他蛇身
            // this.head.angle += 180;
            // this.isFast = true;
            this.setState(AIState.ESCAPING, otherCollider.node?.parent?.getComponent(Snake));
        }
    }

    death() {
        super.death();

        this.node.destroy();

        MainGame.ins.initAiSnake(this.nickName, this.tag);
    }

    // 动态调整AI更新频率
    private updateInterval() {

        const { reactionTime } = this.difficultyParams;

        if (!MainGame.ins.player?.isLife) {
            return reactionTime * 2; // 玩家死亡时降低更新频率
        }

        return reactionTime;

        // const distanceToPlayer = Vec3.distance(
        //     this.head.position,
        //     MainGame.ins.player.head.position
        // );
        //
        // // 距离玩家越近,更新越频繁
        // return math.clamp(
        //     reactionTime + (distanceToPlayer / 1000) * 0.2,
        //     0.05,  // 最小间隔
        //     0.3    // 最大间隔
        // );
    }

    onUpdate(dt: number) {
        if (!this.isLife) return;

        this.updateTimer += dt;
        this.currentUpdateInterval = this.updateInterval();

        if (this.updateTimer >= this.currentUpdateInterval) {
            this.updateTimer = 0;
            this.updateAIState();
        }

        this.executeCurrentState(dt);
        super.onUpdate(dt);
    }

    private updateAIState() {

        // 缓存当前位置
        const player = MainGame.ins.player;

        // 只有在非常危险时才逃跑
        if (this.isInDangerousPosition(this.head.position)) {
            this.setState(AIState.ESCAPING, null);
            return;
        }

        // 检查是否有致命威胁需要逃跑
        const threat = this.findNearestThreat();
        if (threat && this.isLethalThreat(threat)) {
            this.setState(AIState.ESCAPING, threat);
            return;
        }

        // 优先检查是否可以协助其他AI攻击玩家
        const assistPartner = this.findAssistPartner(player);
        if (assistPartner) {
            this.setState(AIState.ASSISTING, assistPartner);
            return;
        }

        // 检查是否可以拦截玩家
        if (this.canInterceptPlayer(player)) {
            this.setState(AIState.INTERCEPTING, player);
            return;
        }

        // 寻找食物
        const nearestFood = this.findNearestFood();
        if (nearestFood) {
            this.setState(AIState.HUNTING, nearestFood);
            return;
        }

        // 默认追击玩家
        if (
            player && player.isLife
            && this.length > player.length * 0.8 // 保持一定实力优势
            && math.random() < this.difficultyParams.aggressiveness * 0.6
            && Vec3.squaredDistance(this.head.position, player.head.position) < this.difficultyParams.viewDisSq // 只在较近距离时考虑追击
        ) { // 追击概率
            this.setState(AIState.INTERCEPTING, player);
        } else {
            this.setState(AIState.WANDERING, null);
        }
    }

    // 判断是否在极度危险的位置（非常靠近边界）
    private isInDangerousPosition(position: Vec3): boolean {
        const dangerBuffer = this.ESCAPE_BOUNDARY;
        const halfWidth = Global.HALF_MAP_WIDTH;
        const halfHeight = Global.HALF_MAP_HEIGHT;

        // 使用 || 短路运算提高效率
        return (Math.abs(position.x) > halfWidth - dangerBuffer ||
            Math.abs(position.y) > halfHeight - dangerBuffer);
    }

    // 判断是否是致命威胁
    private isLethalThreat(threat: Snake): boolean {
        const myPos = this.head.position;
        const threatPos = threat.head.position;
        const disSq = Vec3.squaredDistance(myPos, threatPos);

        return disSq < this.difficultyParams.viewDisSq * 0.5 // 减小威胁判定距离
            && threat.length > this.length * 1.5; // 只有当对方明显比自己强时才逃跑;
    }

    private setState(state: AIState, target: any) {
        // 避免不必要的状态切换
        // if (state === this.currentState) {
        //     if (state === AIState.HUNTING && target === this.targetFood) return;
        //     if (state === AIState.INTERCEPTING && target === this.targetSnake) return;
        //     if (state === AIState.ESCAPING && target === this.escapeTarget) return;
        // }

        this.currentState = state;
        switch (state) {
            case AIState.HUNTING:
                this.targetFood = target as Vec2;
                break;
            case AIState.INTERCEPTING:
                this.targetSnake = target as Snake;
                break;
            case AIState.ESCAPING:
                this.escapeTarget = target as Snake;
                break;
            case AIState.WANDERING:
                this.targetFood = null;
                this.targetSnake = null;
                this.escapeTarget = null;
                break;
            case AIState.ASSISTING:
                this.assistTarget = target as AISnake;
                this.targetSnake = MainGame.ins.player;
                break;
        }
    }

    // 寻找可以协助的AI蛇
    private findAssistPartner(player: Snake): AISnake | null {
        if (!player || !player.isLife) return null;

        const myPos = this.head.position;
        const allAISnakes = MainGame.ins.animalNode.children
            .map(node => node.getComponent(AISnake))
            .filter(snake =>
                snake &&
                snake !== this &&
                snake.isLife &&
                (snake.currentState === AIState.INTERCEPTING || snake.currentState === AIState.ASSISTING)
            );

        const disPlayerSq = Vec3.squaredDistance(myPos, player.head.position);

        for (const aiSnake of allAISnakes) {

            // 确保玩家在较近范围内
            if (disPlayerSq < this.ASSIST_DISTANCE_SQ) {
                continue;
            }

            // 保持一定实力才协助
            if (this.length > player.length * 0.7) {
                continue;
            }

            const disSq = Vec3.squaredDistance(myPos, aiSnake.head.position);

            // 减小协助范围，提高协助条件
            if (disSq < this.ASSIST_DISTANCE_SQ) {
                return aiSnake;
            }
        }

        return null;
    }

    // 执行帮助
    private executeAssisting() {
        if (!this.assistTarget || !this.assistTarget.isLife || !this.targetSnake || !this.targetSnake.isLife) {
            return;
        }

        const playerPos = this.targetSnake.head.position;

        // 计算包围位置：在玩家和协助目标的另一侧
        const angle = this.calculateTargetAngle(playerPos);
        const oppositeAngle = (angle + 180) % 360;

        // 计算包围点
        const surroundDistance = 100;
        const radian = oppositeAngle * Math.PI / 180;
        const surroundX = playerPos.x + Math.cos(radian) * surroundDistance;
        const surroundY = playerPos.y + Math.sin(radian) * surroundDistance;
        const surroundPos = v3(surroundX, surroundY, 0);

        // 移动到包围点
        const targetAngle = this.calculateTargetAngle(surroundPos);
        this.smoothRotateToAngle(targetAngle, this.difficultyParams.turnSpeed);

        this.isFast = true;
    }

    private executeCurrentState(dt: number) {

        // 执行原有状态逻辑
        switch (this.currentState) {
            case AIState.HUNTING:
                this.executeHunting();
                break;
            case AIState.INTERCEPTING:
                this.executeIntercepting();
                break;
            case AIState.ESCAPING:
                this.executeEscaping();
                break;
            case AIState.WANDERING:
                this.executeWandering();
                break;
            case AIState.ASSISTING:
                this.executeAssisting();
                break;
        }
    }

    private canInterceptPlayer(player: Snake): boolean {
        if (!player || !player.isLife) return false;

        const { aggressiveness, interceptDisSq } = this.difficultyParams;
        const myPos = this.head.position;
        const playerPos = player.head.position;

        return this.length > player.length * 1.2
            && math.random() < aggressiveness
            && Vec3.squaredDistance(myPos, playerPos) < interceptDisSq
            && !this.isInDangerousPosition(playerPos);
    }

    private executeHunting() {
        if (!this.targetFood) return;

        const myPos = this.head.position;
        const disSq = Vec2.squaredDistance(myPos as IVec2Like, this.targetFood);
        const targetAngle = this.calculateTargetAngle(this.targetFood);
        const angleDiff = Math.abs(this.head.angle - targetAngle);

        // 根据角度差决定转向策略
        if (angleDiff > 90) {
            // 大角度差时执行快速转向
            this.executeAggressiveTurn(targetAngle);
            this.isFast = false;
        } else {
            // 小角度差时正常追逐
            this.smoothRotateToAngle(targetAngle, this.difficultyParams.turnSpeed * 1.5);
            this.isFast = disSq < this.difficultyParams.viewDisSq / 2;
        }
    }

    // 更激进的转向方法
    private executeAggressiveTurn(targetAngle: number) {
        const currentAngle = this.head.angle;
        let angleDiff = targetAngle - currentAngle;

        // 标准化角度差到 -180 到 180 度范围
        while (angleDiff > 180) angleDiff -= 360;
        while (angleDiff < -180) angleDiff += 360;

        // 使用更大的转向速度
        const turnSpeed = this.difficultyParams.turnSpeed * 2.5;
        this.head.angle += math.clamp(angleDiff, -turnSpeed, turnSpeed);
    }

    private executeIntercepting() {
        if (!this.targetSnake || !this.targetSnake.isLife) return;

        const myPos = this.head.position;
        const nearbyAI = this.findNearbyAIToAvoid();

        const predictedPos = this.predictTargetPosition(this.targetSnake);
        const targetAngle = this.calculateTargetAngle(predictedPos);
        if (nearbyAI) {
            const avoidAngle = this.calculateEscapeAngle(nearbyAI.snake.head.position);
            // 增加躲避权重
            const aiDisSq = Vec3.squaredDistance(myPos, nearbyAI.snake.head.position);
            const avoidWeight = math.clamp(1 - aiDisSq / this.difficultyParams.viewDisSq, 0.4, 0.95); // 增加避让权重
            const finalAngle = this.blendAngles(targetAngle, avoidAngle, avoidWeight);

            this.smoothRotateToAngle(finalAngle, this.difficultyParams.turnSpeed * 1.5);
            this.isFast = false; // 避让时始终降速
        } else {
            // 直接追击时使用更快的转向速度
            this.smoothRotateToAngle(targetAngle, this.difficultyParams.turnSpeed * 1.5);
        }

        this.isFast = true;
    }

    // 混合两个角度
    private blendAngles(angle1: number, angle2: number, weight: number): number {
        // 确保角度在 0-360 范围内
        while (angle1 < 0) angle1 += 360;
        while (angle2 < 0) angle2 += 360;
        while (angle1 >= 360) angle1 -= 360;
        while (angle2 >= 360) angle2 -= 360;

        // 计算角度差
        let diff = angle2 - angle1;
        if (diff > 180) diff -= 360;
        if (diff < -180) diff += 360;

        // 根据权重混合角度
        let result = angle1 + diff * weight;
        while (result < 0) result += 360;
        while (result >= 360) result -= 360;

        return result;
    }

    // 寻找需要躲避的附近AI
    private findNearbyAIToAvoid(): { snake: Snake, dangerLevel: number } | null {
        const myPos = this.head.position;
        const myFuturePos = this.predictFuturePosition(myPos, this.head.angle, this.speed * 2);
        let maxDanger = 0;
        let mostDangerousSnake = null;

        const allSnakes = [...MainGame.ins.animalNode.children, MainGame.ins.player.node]
            .map(node => node.getComponent(Snake))
            .filter(snake => snake && snake !== this && snake.isLife);

        for (const snake of allSnakes) {
            let snakeDanger = 0;

            // 检查身体威胁
            const len = snake.bodyArr.length;
            for (let i = 0; i < len; i++) {
                const bodyPart = snake.bodyArr[i];
                const bodyDisSq = Vec2.squaredDistance(myPos, bodyPart.position);
                const futureDistSq = Vec2.squaredDistance(myFuturePos, bodyPart.position as IVec2Like);

                const bodyAngle = this.calculateTargetAngle(bodyPart.position);
                const angleDiff = Math.abs(this.head.angle - bodyAngle);

                const effectiveCheckDisSq = this.COLLISION_CHECK_DISTANCE_SQ;

                if (bodyDisSq < effectiveCheckDisSq && angleDiff < this.DANGER_ANGLE_THRESHOLD) {
                    const distanceDanger = (effectiveCheckDisSq - bodyDisSq) / effectiveCheckDisSq;
                    const angleDanger = (this.DANGER_ANGLE_THRESHOLD - angleDiff) / this.DANGER_ANGLE_THRESHOLD;
                    const futureDanger = futureDistSq < bodyDisSq ? 2.5 : 1; // 增加未来碰撞的危险系数

                    const partDanger = (distanceDanger * angleDanger * futureDanger) * 120;
                    snakeDanger = Math.max(snakeDanger, partDanger);
                }
            }

            // 更新最危险的蛇
            if (snakeDanger > maxDanger) {
                maxDanger = snakeDanger;
                mostDangerousSnake = snake;
            }
        }

        // 降低触发避让的阈值，使AI更容易进入避让状态
        return maxDanger > 15 ? {
            snake: mostDangerousSnake,
            dangerLevel: maxDanger,
        } : null;
    }

    private predictFuturePosition(currentPos: IVec2Like, angle: number, speed: number): IVec2Like {
        return Vec2.add(v2(), currentPos, this.getVelocity(angle).multiplyScalar(speed));
    }

    private executeEscaping() {
        if (!this.escapeTarget?.isLife) {
            // 如果没有特定的逃离目标，检查并避开所有潜在威胁
            this.avoidAllThreats();
            return;
        }

        const myPos = this.head.position;
        const threatPos = this.escapeTarget.head.position;
        const disSq = Vec3.squaredDistance(myPos, threatPos);

        // 紧急避让判定
        const isEmergency = disSq < Math.pow(this.radius * 3, 2);
        const escapeAngle = this.calculateAvoidanceAngle(this.escapeTarget);

        if (isEmergency) {
            // 紧急情况：直接设置角度
            this.head.angle = escapeAngle;
            this.isFast = false;
        } else {
            // 非紧急情况：快速但平滑地转向
            const angleDiff = escapeAngle - this.head.angle;
            // 标准化角度差到 -180 到 180 度范围
            const normalizedDiff = (angleDiff + 180) % 360 - 180;
            this.head.angle += math.clamp(normalizedDiff, -15, 15);
            this.isFast = disSq > Math.pow(this.radius * 5, 2);
        }
    }

    // 避开所有潜在威胁
    private avoidAllThreats() {
        const myPos = this.head.position;
        const allSnakes = [...MainGame.ins.animalNode.children, MainGame.ins.player.node]
            .map(node => node.getComponent(Snake))
            .filter(snake => snake && snake !== this && snake.isLife);

        let nearestThreatDisSq = Infinity;
        let bestEscapeAngle = this.head.angle;
        let hasThreats = false;

        // 检查所有潜在威胁
        for (const snake of allSnakes) {
            const disSq = Vec3.squaredDistance(myPos, snake.head.position);
            if (disSq < this.COLLISION_CHECK_DISTANCE_SQ) {
                hasThreats = true;
                if (disSq < nearestThreatDisSq) {
                    nearestThreatDisSq = disSq;
                    bestEscapeAngle = this.calculateAvoidanceAngle(snake);
                }
            }
        }

        if (hasThreats) {
            // 有威胁时执行避让
            const isEmergency = nearestThreatDisSq < (this.radius * this.radius) * 3;
            if (isEmergency) {
                this.head.angle = bestEscapeAngle;
                this.isFast = false;
            } else {
                const angleDiff = bestEscapeAngle - this.head.angle;
                const normalizedDiff = (angleDiff + 180) % 360 - 180;
                this.head.angle += math.clamp(normalizedDiff, -15, 15);
                this.isFast = nearestThreatDisSq > (this.radius * this.radius) * 5;
            }
        } else {
            // 没有威胁时检查边界
            this.avoidBoundary();
        }
    }

    // 计算避让角度的方法，增加紧急情况下的处理
    private calculateAvoidanceAngle(threat: Snake): number {
        const myPos = this.head.position;
        const threatPos = threat.head.position;
        const baseEscapeAngle = this.calculateEscapeAngle(threatPos);

        // 检查基础逃生角度是否安全
        if (!this.willHitBoundary(baseEscapeAngle)) {
            return baseEscapeAngle;
        }

        // 如果基础角度不安全，寻找最佳替代角度
        const angles = [
            baseEscapeAngle,
            baseEscapeAngle + 45, baseEscapeAngle - 45,
            baseEscapeAngle + 90, baseEscapeAngle - 90,
            baseEscapeAngle + 135, baseEscapeAngle - 135,
            baseEscapeAngle + 180
        ];

        let bestAngle = baseEscapeAngle;
        let maxSafety = -Infinity;

        for (const angle of angles) {
            if (this.willHitBoundary(angle)) continue;

            const safety = this.evaluateEscapeAngleSafety(angle, threat);
            if (safety > maxSafety) {
                maxSafety = safety;
                bestAngle = angle;
            }
        }

        return bestAngle;
    }

    // 评估逃生角度的安全性
    private evaluateEscapeAngleSafety(angle: number, threat: Snake): number {
        const myPos = this.head.position;
        const futurePos = this.predictFuturePosition(myPos, angle, this.radius * 4);
        let safety = 100;

        // 检查与威胁的距离
        const threatDisSq = Vec2.squaredDistance(futurePos as IVec2Like, threat.head.position);
        safety += threatDisSq;

        // 检查边界距离
        const boundaryDist = this.getDistanceToBoundary(futurePos);
        safety += boundaryDist * 2;

        // 检查与其他蛇的距离
        const allSnakes = [...MainGame.ins.animalNode.children, MainGame.ins.player.node]
            .map(node => node.getComponent(Snake))
            .filter(snake => snake && snake !== this && snake !== threat && snake.isLife);

        for (const snake of allSnakes) {
            const disSq = Vec2.squaredDistance(futurePos, snake.head.position as IVec2Like);
            if (disSq < this.COLLISION_CHECK_DISTANCE_SQ) {
                safety -= (this.COLLISION_CHECK_DISTANCE_SQ - disSq);
            }
        }

        return safety;
    }

    private willHitBoundary(angle: number): boolean {
        const myPos = this.head.position;
        const futurePos = this.predictFuturePosition(myPos, angle, this.radius * 4);
        const boundaryDist = this.getDistanceToBoundary(futurePos);
        return boundaryDist < this.ESCAPE_BOUNDARY;
    }

    // 获取到边界的距离
    private getDistanceToBoundary(position: IVec2Like): number {
        return Math.min(
            Global.HALF_MAP_WIDTH - Math.abs(position.x),
            Global.HALF_MAP_HEIGHT - Math.abs(position.y)
        );
    }

    private executeWandering() {
        // 增加方向改变的概率
        if (math.randomRangeInt(0, 30) == 0) {
            const direction = math.randomRangeInt(0, 3);
            const speed = math.randomRangeInt(1, 3);

            if (direction === DirectionType.LEFT) {
                this.head.angle += speed;
            } else if (direction === DirectionType.RIGHT) {
                this.head.angle -= speed;
            }
        }

        // 减少速度变化的频率
        this.isFast = math.random() < this.difficultyParams.aggressiveness * 0.05;
    }

    private avoidBoundary() {
        const myPos = this.head.position;
        const { HALF_MAP_WIDTH, HALF_MAP_HEIGHT } = Global;
        const boundaryBuffer = this.ESCAPE_BOUNDARY;

        let targetAngle = this.head.angle;
        let isCorner = false;

        // 检查四角区域
        if (myPos.x < -HALF_MAP_WIDTH + boundaryBuffer && myPos.y < -HALF_MAP_HEIGHT + boundaryBuffer) {
            targetAngle = 45; // 右上
            isCorner = true;
        } else if (myPos.x > HALF_MAP_WIDTH - boundaryBuffer && myPos.y < -HALF_MAP_HEIGHT + boundaryBuffer) {
            targetAngle = 135; // 左上
            isCorner = true;
        } else if (myPos.x < -HALF_MAP_WIDTH + boundaryBuffer && myPos.y > HALF_MAP_HEIGHT - boundaryBuffer) {
            targetAngle = 315; // 右下
            isCorner = true;
        } else if (myPos.x > HALF_MAP_WIDTH - boundaryBuffer && myPos.y > HALF_MAP_HEIGHT - boundaryBuffer) {
            targetAngle = 225; // 左下
            isCorner = true;
        } else {
            // 检查边界
            if (myPos.x < -HALF_MAP_WIDTH + boundaryBuffer) {
                targetAngle = 0; // 向右
            } else if (myPos.x > HALF_MAP_WIDTH - boundaryBuffer) {
                targetAngle = 180; // 向左
            }

            if (myPos.y < -HALF_MAP_HEIGHT + boundaryBuffer) {
                targetAngle = 90; // 向上
            } else if (myPos.y > HALF_MAP_HEIGHT - boundaryBuffer) {
                targetAngle = 270; // 向下
            }
        }

        // 如果在角落，锁定方向，避免频繁调整
        if (isCorner) {
            this.smoothRotateToAngle(targetAngle, this.difficultyParams.turnSpeed * 1.5);
        } else {
            this.smoothRotateToAngle(targetAngle, this.difficultyParams.turnSpeed);
        }
    }

    private findNearestFood(): Vec2 | null {
        const myPos = this.head.position;
        let nearestFood = null;
        let { viewDisSq, predictionAccuracy, } = this.difficultyParams;

        let minDisSq = viewDisSq;

        const foods = FoodManger.ins.node.children;
        const boundaryBuffer = this.ESCAPE_BOUNDARY;
        const randomFactor = math.lerp(0.7, 1, predictionAccuracy);

        const { HALF_MAP_WIDTH, HALF_MAP_HEIGHT } = Global;

        for (const food of foods) {
            if (!food.isValid || !food.active) continue;

            const foodPos = food.position;
            const disSq = Vec3.squaredDistance(myPos, foodPos);

            // 检查食物是否靠近墙体
            if (foodPos.x < -HALF_MAP_WIDTH + boundaryBuffer || foodPos.x > HALF_MAP_WIDTH - boundaryBuffer ||
                foodPos.y < -HALF_MAP_HEIGHT + boundaryBuffer || foodPos.y > HALF_MAP_HEIGHT - boundaryBuffer) {
                continue; // 跳过靠近墙体的食物
            }

            if (disSq < minDisSq) {
                if (math.random() < randomFactor) {
                    minDisSq = disSq;
                    nearestFood = foodPos;
                }
            }
        }

        if (nearestFood && predictionAccuracy > 0.7) {
            const competitors = this.findCompetitorsForFood(nearestFood, minDisSq);
            if (competitors.length > 0 && !this.canReachFoodFirst(nearestFood, minDisSq, competitors)) {
                return this.findAlternativeFood(foods, myPos, competitors);
            }
        }

        if (nearestFood) {
            const competitors = this.findCompetitorsForFood(nearestFood, minDisSq);

            if (predictionAccuracy > 0.7) {
                if (competitors.length > 0 && !this.canReachFoodFirst(nearestFood, minDisSq, competitors)) {
                    return this.findAlternativeFood(foods, myPos, competitors);
                }
            }

            const angleToFood = this.calculateTargetAngle(nearestFood);
            const currentAngle = this.head.angle;
            const angleDiff = Math.abs(angleToFood - currentAngle);

            // 如果需要大幅度转向，考虑寻找其他食物
            if (angleDiff > 90 && angleDiff < 270) {
                const alternativeFood = this.findAlternativeFood(foods, myPos, competitors);
                if (alternativeFood) {
                    const altAngleDiff = Math.abs(this.calculateTargetAngle(alternativeFood) - currentAngle);
                    if (altAngleDiff < angleDiff) {
                        nearestFood = alternativeFood;
                    }
                }
            }
        }

        return nearestFood;
    }

    // 寻找替代食物
    private findAlternativeFood(foods: Node[], myPos: Vec3, competitors: Snake[]): Vec2 | null {
        let bestAlternative = null;
        let bestScore = -1;

        const { viewDisSq } = this.difficultyParams;

        for (const food of foods) {
            if (!food.isValid || !food.active) continue;

            const foodPos = food.position;
            const disSq = Vec3.squaredDistance(myPos, foodPos);

            if (disSq > viewDisSq) continue;

            // 计算这个食物的得分（考虑距离和竞争者）
            const score = this.calculateFoodScore(foodPos, disSq, competitors);

            if (score > bestScore) {
                bestScore = score;
                bestAlternative = foodPos;
            }
        }

        return bestAlternative;
    }

    // 计算食物的得分
    private calculateFoodScore(foodPos: Vec3, disSq: number, competitors: Snake[]): number {
        const params = this.difficultyParams;

        // 基础分数（距离越近分数越高）
        let score = 1 - (disSq / params.viewDis);

        // 根据竞争者调整分数
        for (const competitor of competitors) {
            const competitorDisSq = Vec3.squaredDistance(competitor.head.position, foodPos);
            if (competitorDisSq < disSq) {
                // 如果竞争者更近，降低分数
                score *= 0.5;
            }
        }

        // 根据难度添加一些随机性
        score *= math.lerp(0.8, 1.2, math.random() * params.predictionAccuracy);

        return score;
    }

    // 判断是否能在竞争者之前到达食物
    private canReachFoodFirst(foodPos: Vec3, myDisSq: number, competitors: Snake[]): boolean {
        const mySpeed = this.speed * this.moveScale;
        const myTimeToReach = myDisSq / mySpeed;

        for (const competitor of competitors) {
            const competitorPos = competitor.head.position;
            const competitorDisSq = Vec3.squaredDistance(competitorPos, foodPos);
            const competitorSpeed = competitor.speed * (competitor instanceof AISnake ? 2 : 1);
            const competitorTimeToReach = competitorDisSq / competitorSpeed;

            if (competitorTimeToReach < myTimeToReach) {
                return false;
            }
        }

        return true;
    }

    // 寻找同样在追逐食物的竞争者
    private findCompetitorsForFood(foodPos: Vec3, minDisSq: number): Snake[] {
        const competitors: Snake[] = [];
        const allSnakes = [...MainGame.ins.animalNode.children, MainGame.ins.player.node];

        for (const snakeNode of allSnakes) {
            const snake = snakeNode.getComponent(Snake);
            if (snake === this || !snake.isLife) continue;

            const distance = Vec3.squaredDistance(snake.head.position, foodPos);
            // 只考虑距离相近或更近的竞争者
            if (distance <= minDisSq * 1.2) {
                competitors.push(snake);
            }
        }

        return competitors;
    }

    private findNearestThreat(): Snake | null {
        const myPos = this.head.position;
        let nearestThreat = null;

        const allSnakes = [...MainGame.ins.animalNode.children, MainGame.ins.player.node];

        for (const snakeNode of allSnakes) {
            const snake = snakeNode.getComponent(Snake);
            if (snake === this) continue;

            // 检查身体威胁，使用实际的身体半径
            for (let i = 0; i < snake.bodyArr.length; i++) {
                const bodyPart = snake.bodyArr[i];
                const bodyDisSq = Vec3.squaredDistance(myPos, bodyPart.position);

                const threatDisSq = Math.pow(this.radius * this.SAFE_MARGIN, 2);

                if (bodyDisSq < threatDisSq) {
                    nearestThreat = snake;
                    break;
                }
            }
        }

        return nearestThreat;
    }

    private calculateEscapeAngle(threatPos: IVec2Like): number {
        const myPos = this.head.position;
        return math.toDegree(Math.atan2(
            myPos.y - threatPos.y,
            myPos.x - threatPos.x
        ));
    }

    private calculateTargetAngle(targetPos: IVec2Like): number {
        const myPos = this.head.position;
        return math.toDegree(Math.atan2(
            targetPos.y - myPos.y,
            targetPos.x - myPos.x
        ));
    }

    private smoothRotateToAngle(targetAngle: number, turnSpeed: number) {
        const currentAngle = this.head.angle;
        let angleDiff = targetAngle - currentAngle;

        // angleDiff %= 360;
        // 标准化角度差到 -180 到 180 度范围
        while (angleDiff > 180) angleDiff -= 360;
        while (angleDiff < -180) angleDiff += 360;

        // 增加转向速度，减少平滑过渡
        const actualTurnSpeed = turnSpeed * 1.5; // 增加基础转向速度

        // 根据角度差的大小调整转向速度
        const speedMultiplier = Math.abs(angleDiff) > 90 ? 2 : 1.5;
        this.head.angle += math.clamp(
            angleDiff,
            -actualTurnSpeed * speedMultiplier,
            actualTurnSpeed * speedMultiplier
        );
    }

    private predictTargetPosition(target: Snake): Vec2 {
        const targetPos = target.head.position;
        const targetSpeed = target.isFast ? target.speed * 2 : target.speed;

        return target.getVelocity()
            .add2f(targetPos.x, targetPos.y)
            .multiplyScalar(targetSpeed * this.PREDICTION_TIME * this.difficultyParams.predictionAccuracy);

        // return v3(predictX, predictY, 0);
    }

    setDifficulty(level: number) {
        this.difficulty = math.clamp(level, 1, 5);
    }
}