import {
    _decorator, EventKeyboard, Input, input, KeyCode, director, Camera, lerp,
} from "cc";

import { Snake } from "./Snake";
import { Joystick } from "./Components/Joystick";
import { FastBtn } from "./Components/FastBtn";
import { Events } from "./Common/Enums";
import gameStore from "../../store/gameStore";

const { ccclass, property } = _decorator;

@ccclass("Player")
export class Player extends Snake {

    @property(Joystick)
    joystick: Joystick = null;

    @property(FastBtn)
    fastBtn: FastBtn = null;

    @property(Camera)
    camera: Camera = null;

    get scale() {
        return super.scale;
    }

    set scale(value: number) {
        super.scale = value;
        this.camera.orthoHeight = lerp(275, 612, value);
    }

    get length() {
        return gameStore.gameInfo.length;
    }

    set length(value: number) {
        gameStore.gameInfo.length = value;
    }


    get luckNum() {
        return gameStore.gameInfo.luckNum;
    }

    set luckNum(value: number) {
        gameStore.gameInfo.luckNum = value;
    }

    onLoad() {
        super.onLoad();
        this.tag = Snake.getTag();
        input.on(Input.EventType.KEY_DOWN, this.onKeyDown, this);
        input.on(Input.EventType.KEY_UP, this.onKeyUp, this);
        this.fastBtn.node.on("fast", this.onFast, this);
    }

    onDestroy() {
        input.off(Input.EventType.KEY_DOWN, this.onKeyDown, this);
        input.off(Input.EventType.KEY_UP, this.onKeyUp, this);
        this.fastBtn?.node?.off("fast", this.onFast, this);
    }

    onFast(isFast: boolean) {
        this.isFast = isFast;
    }

    revive() {
        this.isLife = true;
        this.init({
            tag: this.tag,
            initEnergy: this.energy,
            skinName: this.skinName,
            nickName: this.nickName,
        });
    }

    death() {
        super.death();

        this.length = 0;
        // 发送游戏结束事件
        director.emit(Events.Death);
    }

    keyArr: number[] = [];

    onKeyDown(event: EventKeyboard) {

        const keyArr = [KeyCode.KEY_W, KeyCode.KEY_S, KeyCode.KEY_A, KeyCode.KEY_D,];

        if (keyArr.indexOf(event.keyCode) > -1) {
            this.keyArr.push(event.keyCode);
        }

        if (event.keyCode == KeyCode.SPACE) {
            this.fastBtn.isFast = true;
        }

        this.setDir();
    }

    onKeyUp(event: EventKeyboard) {
        const index = this.keyArr.indexOf(event.keyCode);
        if (index > -1) {
            this.keyArr.splice(index, 1);
        }

        if (event.keyCode == KeyCode.SPACE) {
            this.fastBtn.isFast = false;
        }
        this.setDir();
    }

    setDir() {
        if (!this.keyArr.length) return;
        this.joystick.angle = {
            [KeyCode.KEY_W]: 270,
            [KeyCode.KEY_S]: 90,
            [KeyCode.KEY_A]: 180,
            [KeyCode.KEY_D]: 0,
        }[this.keyArr[this.keyArr.length - 1]];
    }

    onUpdate(dt: number) {
        this.head.angle = (360 - this.joystick.angle) % 360;
        super.onUpdate(dt);
    }

}
