import { _decorator, Enum, SpriteFrame, math, v3, Sprite, Collider2D, Component, Prefab, ccenum } from "cc";
import { PoolManager } from "db://assets/Scripts/Scenes/MainGame/Manager/PoolManager";
import { FoodType } from "db://assets/Scripts/Scenes/MainGame/Common/Enums";
import { Global } from "db://assets/Scripts/Scenes/MainGame/Global";
import { executePreFrame, getItemGenerator } from "db://assets/Scripts/Utils/ExecutePreFrame";

const { ccclass, property } = _decorator;

/**
 * 食物项配置
 */
@ccclass("Item")
class Item {
    @property({ type: FoodType })
    type: FoodType = FoodType.FOOD;

    @property(SpriteFrame)
    sp: SpriteFrame = null;
}

/**
 * 食物管理器
 */
@ccclass("FondManger")
export class FondManger extends Component {

    @property({ type: [Item] })
    private foods: Item[] = [];

    @property(Prefab)
    private foodPrefab: Prefab = null;

    maxFood: number = 100;

    /**
     * 获取当前食物数量
     */
    getFoodSum() {
        return this.node.children.length || 0;
    }

    /**
     * 初始化食物
     */
    async init(maxFood: number = 100) {
        this.maxFood = maxFood;

        await this.initFond(maxFood);

        this.schedule(this.checkFood, 1);
    }

    /**
     * 检查食物
     */
    checkFood() {
        for (let i = this.getFoodSum(); i < this.maxFood; i++) {
            this.addFood();
        }
    }

    /**
     * 设置食物
     */
    addFood(x?: number, y?: number) {

        // 如果没有指定位置,随机生成位置
        if (!x) {
            x = math.randomRangeInt(-(Global.MAP_WIDTH / 2 - 50), Global.MAP_WIDTH / 2 - 50);
        }

        if (!y) {
            y = math.randomRangeInt(-(Global.MAP_HEIGHT / 2 - 50), Global.MAP_HEIGHT / 2 - 50);
        }

        // 从对象池获取食物节点
        const node = PoolManager.instance.getNode(this.foodPrefab);

        // 随机选择食物类型
        const index = math.randomRangeInt(0, this.foods.length);

        this.node.addChild(node);
        // 设置食物属性
        node.angle = math.randomRange(0, 360);
        node.setScale(1, 1);
        node.setPosition(x, y);
        node.getComponent(Sprite).spriteFrame = this.foods[index].sp;

        // 设置碰撞类型
        const collider = node.getComponent(Collider2D);
        if (collider) {
            collider.tag = this.foods[index].type;
        }

        node.active = true;
    }

    /**
     * 初始化单个食物
     */
    initItem = (_: number) => {
        // 随机生成位置
        const x = math.randomRangeInt(-(Global.MAP_WIDTH / 2 - 50), Global.MAP_WIDTH / 2 - 50);
        const y = math.randomRangeInt(-(Global.MAP_HEIGHT / 2 - 50), Global.MAP_HEIGHT / 2 - 50);

        this.addFood(x, y);
    };

    /**
     * 初始化食物
     */
    async initFond(count: number) {
        await executePreFrame(getItemGenerator(count, this.initItem), 1, this);
    }

}
