import {
    _decorator,
    Camera,
    director,
    Label,
    math,
    Node,
    PhysicsSystem2D,
    Prefab,
    UITransform,
} from "cc";
import { Joystick } from "./Components/Joystick";
import { FastBtn } from "./Components/FastBtn";
import { FondManger } from "./Manager/FondManger";
import { Global } from "./Global";
import { Events, GameState } from "./Common/Enums";
import { PoolManager } from "./Manager/PoolManager";
import { showToast } from "../../../Module/UIFast";
import Scene from "../../../Module/Scene";
import { executePreFrame, getItemGenerator } from "../../Utils/ExecutePreFrame";
import { Player } from "./Player";
import { AISnake } from "./AISnake";

const { ccclass, property } = _decorator;


@ccclass("MainGame")
export class MainGame extends Scene {

    static bundle: string = "MainGame";
    static skin: string = "MainGame";

    @property(Prefab)
    private animalPrefab: Prefab = null;

    @property({
        displayName: "最多食物",
        tooltip: "地图上随机产生食物，如果超过当前值不在产生。"
    })
    maxFood: number = 200;

    @property({
        displayName: "NPC数量",
        tooltip: "当前游戏最大NPC角色"
    })
    private maxAnimal: number = 20;

    @property({ type: Joystick, displayName: "摇杆" }) joystick: Joystick = null;

    @property({
        type: FastBtn,
        displayName: "快进按钮"
    })
    private fastBtn: FastBtn = null;

    @property(UITransform)
    private uiBg: UITransform = null;

    @property(Player)
    player: Player = null;

    @property(FondManger)
    fondManger: FondManger = null;

    @property(Node)
    animalNode: Node = null;

    @property(Label)
    private LTips: Label = null;

    @property(Camera)
    camera: Camera = null;

    private state: GameState = GameState.READY;
    private rebirthSum: number = 0;

    private static _ins: MainGame = null;
    static get ins(): MainGame {
        return MainGame._ins;
    }

    async onLoad() {
        MainGame._ins = this;

        PhysicsSystem2D.instance.enable = true;
        // PhysicsSystem2D.instance.gravity = math.Vec2.ZERO;
        // PhysicsSystem2D.instance.debugDrawFlags = EPhysics2DDrawFlags.Aabb |
        //     EPhysics2DDrawFlags.Pair |
        //     EPhysics2DDrawFlags.CenterOfMass |
        //     EPhysics2DDrawFlags.Joint |
        //     EPhysics2DDrawFlags.Shape;

        Global.MAP_WIDTH = this.uiBg.contentSize.x;
        Global.MAP_HEIGHT = this.uiBg.contentSize.y;

        // 初始化墙壁
        const wallTop = this.uiBg.node.getChildByName("WALL_TOP");
        wallTop.setPosition(0, Global.MAP_HEIGHT / 2);
        wallTop.getComponent(UITransform).width = Global.MAP_WIDTH;

        const wallBottom = this.uiBg.node.getChildByName("WALL_BOTTOM");
        wallBottom.setPosition(0, -Global.MAP_HEIGHT / 2);
        wallBottom.getComponent(UITransform).width = Global.MAP_WIDTH;

        const wallLeft = this.uiBg.node.getChildByName("WALL_LEFT");
        wallLeft.setPosition(-Global.MAP_WIDTH / 2, 0);
        wallLeft.getComponent(UITransform).height = Global.MAP_HEIGHT;

        const wallRight = this.uiBg.node.getChildByName("WALL_RIGHT");
        wallRight.setPosition(Global.MAP_WIDTH / 2, 0);
        wallRight.getComponent(UITransform).height = Global.MAP_HEIGHT;

        this.player.init();

        // 初始化食物和NPC
        this.fondManger.init(this.maxFood);
        this.initAnimal(this.maxAnimal);

        // 设置游戏状态
        this.setGameState(GameState.PLAY);

        // 注册事件
        director.on(Events.showGOver, this.showGOver, this);
        director.on(Events.setGameState, this.setGameState, this);
    }

    onDestroy() {
        MainGame._ins = null;
    }

    update(dt: number) {
        if (this.state == GameState.READY) return;

        // 更新UI提示
        this.LTips.string = `长度：${this.player.getSnakeLen()}}`;

        this.player.onUpdate(dt);

        // 更新相机位置
        this.camera.node.setPosition(this.player.head.getPosition());

        // 更新NPC移动
        this.animalNode.children.forEach(child => {
            child.getComponent(AISnake)?.onUpdate(dt);
        });
    }

    onPause() {
        this.setGameState(GameState.PAUSE);
        console.log("pause");
    }

    setGameState(state: GameState) {
        this.state = Number(state);
        switch (this.state) {
            case GameState.READY:
                break;
            case GameState.PLAY:
                director.resume();
                break;
            case GameState.PAUSE:
                director.pause();
                break;
            case GameState.OVER:
                this.rebirthSum++;
                if (this.rebirthSum > 2) return this.showGOver();
                showToast("你已死亡！");
                break;
            case GameState.WIN:
                director.pause();
                console.log("win", this.player.getSnakeLen());
                break;
            case GameState.QUIT:
                director.resume();
                director.loadScene("GameMain");
                break;
            default:
                console.log("err");
        }
    }

    showGOver() {
        console.log("showGOver", this.player.getSnakeLen());
    }

    initItem = (index: number) => {
        const node = PoolManager.instance.getNode(this.animalPrefab);
        const x = math.randomRangeInt(-(Global.MAP_WIDTH / 2 - 50), Global.MAP_WIDTH / 2 - 50);
        const y = math.randomRangeInt(-(Global.MAP_HEIGHT / 2 - 50), Global.MAP_HEIGHT / 2 - 50);

        node.getComponent(AISnake)?.init({
            x, y,
            angle: math.randomRangeInt(0, 360),
            skinName: "default",
        });

        this.animalNode.addChild(node);
    };

    async initAnimal(count: number) {
        await executePreFrame(getItemGenerator(count, this.initItem), 1, this);
    }

}