import { _decorator } from "cc";
import { BaseBuff } from "./BaseBuff";
import { BuffType } from "./BuffType";
import { Snake } from "../Snake";
import { ShieldBuff, SpeedUpBuff, MagnetBuff } from "./Buffs";

const { ccclass } = _decorator;

@ccclass("BuffManager")
export class BuffManager {
    private buffs: Map<BuffType, BaseBuff> = new Map();
    private target: Snake = null;

    constructor(target: Snake) {
        this.target = target;
    }

    // 添加Buff
    addBuff(buff: BaseBuff): void {
        const type = buff.getType();
        
        // 如果已存在同类型Buff，先移除
        if (this.buffs.has(type)) {
            this.removeBuff(type);
        }

        this.buffs.set(type, buff);
        buff.activate(this.target);
    }

    // 移除指定类型的Buff
    removeBuff(type: BuffType): void {
        const buff = this.buffs.get(type);
        if (buff) {
            buff.deactivate();
            this.buffs.delete(type);
        }
    }

    // 更新所有Buff
    update(dt: number): void {
        this.buffs.forEach((buff, type) => {
            if (!buff.update(dt)) {
                this.buffs.delete(type);
            }
        });
    }

    // 检查是否有某个Buff
    hasBuff(type: BuffType): boolean {
        return this.buffs.has(type);
    }

    // 获取Buff
    getBuff(type: BuffType): BaseBuff | null {
        return this.buffs.get(type) || null;
    }

    // 清除所有Buff
    clearAll(): void {
        this.buffs.forEach(buff => buff.deactivate());
        this.buffs.clear();
    }

    // 获取所有激活的Buff
    getActiveBuffs(): BaseBuff[] {
        return Array.from(this.buffs.values());
    }

    // 创建Buff的便捷方法
    createBuff(type: BuffType, duration?: number, value?: number): BaseBuff {
        switch (type) {
            case BuffType.SHIELD:
                return new ShieldBuff(duration);
            case BuffType.SPEED_UP:
                return new SpeedUpBuff(duration, value);
            case BuffType.MAGNET:
                return new MagnetBuff(duration, value);
            default:
                return null;
        }
    }
} 