import { _decorator } from "cc";
import { BuffType, BuffState } from "./BuffType";
import { Snake } from "../Snake";

const { ccclass, property } = _decorator;

@ccclass("BaseBuff")
export class BaseBuff {
    protected type: BuffType = BuffType.NONE;
    protected duration: number = 0;
    protected remainTime: number = 0;
    protected value: number = 0;
    protected state: BuffState = BuffState.INACTIVE;
    protected target: Snake = null;
    
    constructor(type: BuffType, duration: number, value: number) {
        this.type = type;
        this.duration = duration;
        this.remainTime = duration;
        this.value = value;
    }

    // 激活Buff
    activate(target: Snake): void {
        this.target = target;
        this.state = BuffState.ACTIVE;
        this.onActivate();
    }

    // 更新Buff状态
    update(dt: number): boolean {
        if (this.state !== BuffState.ACTIVE) return false;

        this.remainTime -= dt;
        this.onUpdate(dt);

        if (this.remainTime <= 0) {
            this.deactivate();
            return false;
        }

        return true;
    }

    // 停用Buff
    deactivate(): void {
        this.state = BuffState.FINISHED;
        this.onDeactivate();
    }

    // 获取剩余时间
    getRemainTime(): number {
        return Math.max(0, this.remainTime);
    }

    // 获取Buff类型
    getType(): BuffType {
        return this.type;
    }

    // 获取Buff状态
    getState(): BuffState {
        return this.state;
    }

    // 获取Buff值
    getValue(): number {
        return this.value;
    }

    // 以下是子类需要实现的方法
    protected onActivate(): void { }
    protected onUpdate(dt: number): void { }
    protected onDeactivate(): void { }
} 