import { Vec2, Vec3, Size, assetManager, SpriteFrame, SpriteAtlas } from "cc";

/**
 * 判断两点是否在指定范围内相交
 * @param pos1 位置1
 * @param pos2 位置2
 * @param rangeW 宽度范围
 * @param rangeH 高度范围
 * @returns 是否相交
 */
export function isIntersect(pos1: Vec3, pos2: Vec3, rangeW: number, rangeH: number): boolean {
  return rangeW > Math.abs(pos1.x - pos2.x) && rangeH > Math.abs(pos1.y - pos2.y);
}

/**
 * 获取随机浮点数
 * @param min 最小值
 * @param max 最大值
 * @returns 随机浮点数
 */
export function getRandomFloat(min: number, max: number): number {
  return Math.random() * (max - min) + min;
}


/**
 * 获取随机位置
 * @param viewSize 视图大小
 * @param padding 边距
 * @returns 随机位置
 */
export function getRandomPosition(viewSize: Size, padding: number = 0): Vec2 {
  const x = getRandomFloat(
    -viewSize.width / 2 + padding,
    viewSize.width / 2 - padding
  );
  const y = getRandomFloat(
    -viewSize.height / 2 + padding,
    viewSize.height / 2 - padding
  );
  return new Vec2(x, y);
}


export function formatTime(seconds: number): string {
  const mins = Math.floor(seconds / 60);
  const secs = Math.floor(seconds % 60);
  return `${mins.toString().padStart(2, "0")}:${secs.toString().padStart(2, "0")}`;
}

export function loadSkin(skinName: string, bundleName: string = "MainGame"): Promise<SpriteAtlas> {
    return new Promise((resolve, reject) => {
        assetManager.getBundle(bundleName).load(`skin/${skinName}`, SpriteAtlas,
            (err, skin) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(skin);
            }
        );
    });

}