import { _decorator, Color, Component, color, ImageAsset, Sprite, SpriteFrame, Texture2D, UITransform } from 'cc';
import HtmlToImage from "html-to-image";

const {ccclass, property, requireComponent} = _decorator;

@ccclass('HTMLRichText')
@requireComponent(Sprite)
export class HTMLRichText extends Component {

    static _id = 0;

    static EventType = {
        AFTER_RENDER: "AFTER_RENDER",
    }

    _html: string = "";

    @property({multiline: true})
    set html(val: string) {
        this._html = val;
        this.setTxt(val);
    }

    get html(): string {
        return this._html;
    }

    @property
    fontSize: number = 20;

    @property(Color)
    color: Color = color(0, 0, 0, 255);

    protected start() {
    }

    imgAssets: ImageAsset = null;
    t2d: Texture2D = new Texture2D();
    sf: SpriteFrame = new SpriteFrame();


    async setTxt(html: string) {

        if (!html) {
            this.node.getComponent(Sprite).spriteFrame = null;
            return;
        }

        const id = `richTxt_${HTMLRichText._id++}`;

        const width = this.node.getComponent(UITransform).width;
        const color = this.color.toCSS("rgba");

        const dom = document.createElement("div");
        dom.innerHTML = html;
        dom.id = id;
        dom.style.position = "fixed";
        dom.style.zIndex = "-9999";
        dom.style.width = `${width}px`;
        dom.style.fontSize = `${this.fontSize}px`;
        dom.style.color = color;
        document.body.appendChild(dom);

        const canvas = await HtmlToImage.toCanvas(dom, {
            skipFonts: true,
            backgroundColor: "transparent",
            // pixelRatio: devicePixelRatio,
        }).finally(() => {
            document.body.removeChild(dom);
        });

        if (!canvas) return;

        this.imgAssets = new ImageAsset(canvas);

        this.t2d.image = this.imgAssets;
        this.sf.texture = this.t2d;

        this.node.getComponent(Sprite).spriteFrame = this.sf;

        this.node.getComponent(UITransform).setContentSize(canvas.width, canvas.height);

        this.node.emit(HTMLRichText.EventType.AFTER_RENDER);

    }

    onDestroy() {
        this.imgAssets?.destroy();
        this.t2d.destroy();
        this.sf.destroy();
    }

}


