import { Base, IBase } from "@/pages/HomePage/Top/base/Base.ts";
import { Assets, Container, Sprite, Text, Ticker } from "pixi.js";
import { Ease, Tween } from "@/core/tween";
import store from "@/store/store.ts";
import { _asyncThrottle, prefixInteger } from "@/utils/utils.ts";
import API from "@/api";
import { GameEvent, globalEvent } from "@/pages/HomePage/Top/GameEvent.ts";
import { BubbleMgr } from "@/pages/HomePage/Top/Components/BubbleMgr.ts";

export interface IBubbleInfo {
  id: string;
  creditsNum: string,
  expireTime: number,
  type: "1" | "2",  // 1-签到气泡 2-积分气泡
}

export class Bubble extends Container implements IBase {
  text: Text;
  time: Text;
  id: string;

  constructor() {
    super();
    this.onLoad();
    this.on("pointerup", this.onPointerUp, this);
  }

  onLoad() {
    this.addChild(new Sprite(Assets.get("黄色气泡.png")))
      .anchor.set(0.5);

    this.text = this.addChild(new Text({
      text: `210`,
      style: {
        fontSize: 42,
        fill: 0xffffff,
        align: "center",
        fontWeight: "bold",
      }
    }));
    this.text.anchor.set(0.5);

    this.time = this.addChild(new Text({
      text: `23:59:59`,
      style: {
        fontSize: 21,
        fill: 0xffffff,
        align: "center",
        fontWeight: "bold",
      }
    }));
    this.time.anchor.set(0.5);
    this.time.position.set(0, 65);
  }

  onPointerUp = _asyncThrottle(async () => {
    const { x, y, parent } = this;

    const { bubbleArr, bubbleMap } = parent as BubbleMgr;
    const i = bubbleArr.indexOf(this);
    if (i !== -1) bubbleArr[i] = null;
    bubbleMap.delete(this.id);

    this.removeFromParent();
    this.destroy();

    const { success, data } = await API.receiveBubble({ id: this.id });
    store.updateIndexThrottle();
    if (success) {
      globalEvent.emit(GameEvent.ReceiveEffect, { x, y });
    }
  });

  ideaEffect(wait: number, dir = 1) {
    const y = this.y;
    Tween.get(this)
      .wait(wait)
      .call(() => {
        Tween.get(this, { loop: true })
          .to({ y: y + 25 * dir }, 2000, Ease.quadInOut)
          .to({ y: y }, 2000, Ease.quadInOut);
      });
  }

  cdTarget = { cd: 0 }

  setInfo(info: IBubbleInfo) {
    const { id, creditsNum, expireTime, type } = info;
    this.id = id;
    this.text.text = creditsNum;

    this.cdTarget.cd = Math.max((expireTime - Date.now()) / 1000 >> 0, 1);
    this.setTime();
    Tween.removeTweens(this.cdTarget);
    Tween.get(this.cdTarget, { loop: true })
      .wait(1000)
      .call(() => {
        this.cdTarget.cd -= 1;
        if (this.cdTarget.cd <= 0) {
          this.cdTarget.cd = 0;
          store.updateIndexThrottle();
          Tween.removeTweens(this.cdTarget);
        }
        this.setTime();
      });
  }

  setTime() {
    const cd = this.cdTarget.cd;
    const h = Math.floor(cd / 3600);
    const min = Math.floor((cd - h * 3600) / 60);
    const sec = Math.floor(cd - h * 3600 - min * 60);
    this.time.text = `${prefixInteger(h, 2)}:${prefixInteger(min, 2)}:${prefixInteger(sec, 2)}`;
  }

  onUpdate(time: Ticker) {

  }

  onDestroy() {
    Tween.removeTweens(this);
    Tween.removeTweens(this.cdTarget);
  }

  destroy() {
    this.onDestroy();
    super.destroy();
  }

}
