import { BuildHook, IBuildResult, IBuildTaskOption } from '../@types';
import * as fs from "fs";
import { exec } from "child_process";
import AutoUpload from "./upload";
import { obfuscate } from "./obfuscator";
import { compressAllImage } from "./minImg";
import { zipDir } from "./zip";

interface IOptions {
    useZip: boolean;
    useJsEncryption: boolean;
    uploadDir: string;
    bucket: string;
    accessKeyId: string;
    accessKeySecret: string;
    region: string;
    cdnDomain: string;
}

const PACKAGE_NAME = 'duiba-publish';

interface ITaskOptions extends IBuildTaskOption {
    packages: {
        [PACKAGE_NAME]: IOptions;
    };
}


let allAssets = [];

export const throwError: BuildHook.throwError = true;

export const load: BuildHook.load = async function () {
    console.log(`[${PACKAGE_NAME}] Load cocos plugin example in builder.`);
    allAssets = await Editor.Message.request('asset-db', 'query-assets');
};


let buildVersion = 0;

function getRemotePath(uploadDir = 'template3d') {
    return `/${uploadDir}/${buildVersion}/`;
}

function getRemoteUrl(cdnDomain: string, uploadDir = 'template3d') {
    // return `https://yun.duiba.com.cn${getRemotePath(uploadDir)}`;
    // https://duiba-credits-test.oss-cn-hangzhou.aliyuncs.com/dc.png
    // https://yun-duiba-credits-test.duibatest.com.cn/
    // console.log('cdnDomain', cdnDomain);
    console.log("%c兑吧发布插件 >> 变态代码防调试,cdnDomain: " + cdnDomain, "color: green");
    return `${cdnDomain}${getRemotePath(uploadDir)}`;
}


function getGitInfo(path) {
    return new Promise((resolve) => {
        let gitStr = '';
        exec(`cd ${path} && git remote -v && git branch --show-current & git config --global user.name`, (_err, stdout, _stderr) => {
            if (stdout) gitStr += encodeURIComponent(stdout.replace(/\n/g, ';'));
            resolve(gitStr);
        });

    });
}

export const onBeforeBuild: BuildHook.onBeforeBuild = async function (options: ITaskOptions, result: IBuildResult) {
    if (options.platform !== "web-mobile") return;

    buildVersion = Date.now();
};

export const onBeforeCompressSettings: BuildHook.onBeforeCompressSettings = async function (options: ITaskOptions, result: IBuildResult) {
    if (options.platform !== "web-mobile") return;
    const remoteUrl = getRemoteUrl(options.packages[PACKAGE_NAME].cdnDomain, options.packages[PACKAGE_NAME].uploadDir);
    result.settings.assets.server = remoteUrl;
    result.settings.rendering.effectSettingsPath = getRemoteUrl(options.packages[PACKAGE_NAME].cdnDomain, options.packages[PACKAGE_NAME].uploadDir) + 'src/effect.bin';
    result.settings.plugins.jsList = [];
    result.settings.splashScreen = {
        displayRatio: 1,
        totalTime: 0,
        watermarkLocation: "default",
        autoFit: true,
    };
    // result.settings.plugins.jsList = result.settings.plugins.jsList.map((jsUrl) => {
    //    return `${remoteUrl}src/${jsUrl}`;
    // });
};

export const onAfterCompressSettings: BuildHook.onAfterCompressSettings = async function (options: ITaskOptions, result: IBuildResult) {
    // Todo some thing
    console.log('webTestOption', 'onAfterCompressSettings');
};

export const onAfterBuild: BuildHook.onAfterBuild = async function (options: ITaskOptions, result: IBuildResult) {
    if (options.platform !== "web-mobile") return;

    const {
        useZip, useJsEncryption,
        uploadDir, accessKeySecret, accessKeyId, bucket, region,
    } = options.packages[PACKAGE_NAME];

    const remoteUrl = getRemoteUrl(options.packages[PACKAGE_NAME].cdnDomain, options.packages[PACKAGE_NAME].uploadDir);

    /*************************** 防调试 ***************************/
    // 非调试模式下开启变态代码防调试
    if (useJsEncryption && !options.debug) {
        console.log("%c兑吧发布插件 >> 变态代码防调试", "color: green");
        if (fs.existsSync(result.dest + "/remote")) {
            fs.readdirSync(result.dest + "/remote")
                .forEach((dirName) => {
                    if (dirName !== "main") {
                        return;
                    }
                    obfuscate(result.dest + "/remote/" + dirName + "/index.js");
                });
        }

        if (fs.existsSync(result.dest + "/assets")) {
            fs.readdirSync(result.dest + "/assets")
                .forEach((dirName) => {
                    if (dirName !== "main") {
                        return;
                    }
                    obfuscate(result.dest + "/assets/" + dirName + "/index.js");
                });
        }

        if (fs.existsSync(result.dest + "/src/chunks")) {
            fs.readdirSync(result.dest + "/src/chunks")
                .forEach((fileName) => {
                    obfuscate(result.dest + "/src/chunks/" + fileName);
                });
        }
        console.log("%c兑吧发布插件 >> 变态代码防调试成功", "color: green");
    } else {
        console.warn("兑吧发布插件 >> 已开启调试模式，请确保不是上线代码");
        console.warn("兑吧发布插件 >> 已开启调试模式，请确保不是上线代码");
        console.warn("兑吧发布插件 >> 已开启调试模式，请确保不是上线代码");
        await Editor.Dialog.warn("兑吧发布插件\n已开启调试模式\n请确保不是上线代码");
    }
    /*************************** 防调试 ***************************/


    /*************************** 修改脚本 ***************************/
    console.log("%c兑吧发布插件 >> 开始修改脚本", "color: green");
    // application.js 所在路径
    const mainJsPath = result.dest + "/application.js";

    // 读取 application.js
    let script = fs.readFileSync(mainJsPath, "utf8");

    const gitInfo = await getGitInfo(result.dest);
    // 添加一点脚本
    script = `window.__remoteUrl__ = \"${remoteUrl}\";\n`
        + `window.__remoteAssets__ = window.__remoteUrl__ + \"assets/\";\n`
        + `window.__version__ = ${buildVersion};\n`
        + `window.__ENV__ = "prod";\n`
        + `window.___G___ = "${gitInfo}";`
        + script;

    if (useZip) {
        console.log("%c兑吧发布插件 >> 使用zip", "color: green")
        script =
            `window.zipBundle = ${JSON.stringify(result.settings.assets.remoteBundles)};`
            + script;
    }

    script = script.replace(
        "src/settings.json",
        `${remoteUrl}src/settings.json`
    );

    // 保存
    fs.writeFileSync(mainJsPath, script);

    console.log("%c兑吧发布插件 >> 修改脚本完成", "color: green");
    /*************************** 修改脚本 ***************************/


    /*************************** 压缩图片 ***************************/
    console.log("%c兑吧发布插件 >> 开始压缩图片", "color: green");

    await compressAllImage(["assets", "remote"], result.dest);

    console.log("%c兑吧发布插件 >> 压缩图片结束", "color: green");
    /*************************** 压缩图片 ***************************/


    /*************************** 生成皮肤模版 ***************************/
    console.log("%c兑吧发布插件 >> 开始生成皮肤模版", "color: green");

    const htmlPath = result.dest + "/index.html";

    const indexHtml = fs.readFileSync(htmlPath, "utf8")
        .replace(/{{__remoteUrl__}}/g, remoteUrl);

    fs.writeFileSync(htmlPath, indexHtml);

    console.log("%c兑吧发布插件 >> 生成皮肤模版完成", "color: green");
    /*************************** 生成皮肤模版 ***************************/

    /*************************** 压缩资源 ***************************/
    if (useZip) {
        console.log("%c兑吧发布插件 >> 使用zip", "color: green")
        if (fs.existsSync(result.dest + "/remote")) {
            await Promise.all(
                fs.readdirSync(result.dest + "/remote")
                    .map((dirName) => {
                        return zipDir(dirName, result.dest + "/remote/" + dirName, result.dest + "/remote");
                    })
            )
        }
    }
    /*************************** 压缩资源 ***************************/

    /*************************** 打包完成，开始上传 ***************************/
    console.log("%c兑吧发布插件 >> 打包完成，开始上传", "color: green");

    const autoUpload = new AutoUpload({
        dir: result.dest,
        originDir: getRemotePath(uploadDir),
        accessKeySecret, accessKeyId, bucket, region,
    });

    await autoUpload.start();

    console.log("%c兑吧发布插件 >> 上传完成", "color: green")
    /*************************** 打包完成，开始上传 ***************************/


    console.log("%c兑吧发布插件 >> 版本号:" + buildVersion, "color: green");
    await Editor.Dialog.info("兑吧发布插件\n上传完成\n可以发布");
};

export const unload: BuildHook.unload = async function () {
    console.log(`[${PACKAGE_NAME}] Unload cocos plugin example in builder.`);
};

export const onError: BuildHook.onError = async function (options, result) {
    // Todo some thing
    console.warn(`${PACKAGE_NAME} run onError`);
};

export const onBeforeMake: BuildHook.onBeforeMake = async function (root, options) {
    console.log(`onBeforeMake: root: ${root}, options: ${options}`);
};

export const onAfterMake: BuildHook.onAfterMake = async function (root, options) {
    console.log(`onAfterMake: root: ${root}, options: ${options}`);
};
