import { makeAutoObservable } from "mobx"; // 从mobx库导入makeAutoObservable，用于自动观察类的状态变化
import { playSound, preloadSounds, registerSounds, stopSound, watchPageVisibility, mute, unmute } from "@spark/utils"; // 从工具库导入多个声音控制的函数
import { RES_PATH } from "../../sparkrc"; // 导入资源路径常量

// 创建一个音乐存储的类并使用mobx的makeAutoObservable进行自动观察
const musicStore = makeAutoObservable(new class {

  // 定义音乐状态，开发环境下默认不播放音乐，初始值为true（音乐播放）
  status = true;

  constructor() {
    // 注册背景音乐的声音资源
    registerSounds({
      'bg_music': RES_PATH + 'music/bgm.mp3', // 将背景音乐的路径与名称进行关联
    });

    // 预加载声音资源
    preloadSounds(null, () => {
      // 加个判断 防止出现：音乐还未加载完用户点击了暂停，等加载完音乐还会播放
      musicStore.status && playSound('bg_music', { 'loop': true }); // 如果状态为true，则循环播放背景音乐
    });

    // 监听页面可见性变化
    watchPageVisibility(this.onPageVisibilityChange);
  }

  // 页面可见性变化的处理函数
  onPageVisibilityChange = async (visible) => {
    if (visible) {
      console.info('页面可见',this.status);
      // 如果页面可见，判断当前状态决定是否播放背景音乐
      this.status ? await playSound('bg_music', {'loop': true}) : stopSound('bg_music');
    } else {
      // 如果页面不可见，停止播放背景音乐
      stopSound('bg_music');
      console.info('页面不可见',this.status);
    }
  }
  // 切换音乐状态的方法
  changeMusicStatus () {
    // 切换当前状态, 播放或停止音乐, 添加默认淡入淡出效果
    this.status = !this.status;
    this.status ? playSound('bg_music', { 'loop': true, 'fade': true }) : stopSound('bg_music', { 'fade': true });
  }
});

// 导出音乐存储实例，以便在其他模块使用
export default musicStore;
