package cn.com.duiba.spring.boot.starter.dsp.rateLimiter;

import com.google.common.collect.Lists;
import lombok.Data;
import java.util.List;

/**
 * 令牌桶限流器的执行对象
 */
@Data
public class TokenBucketLimiterPolicy {

    /**
     * 限流时间间隔
     * (重置桶内令牌的时间间隔)
     */
    private long resetBucketInterval;
    /**
     * 最大令牌数量
     */
    private long bucketMaxTokens;

    /**
     * 初始可存储数量
     */
    private long initTokens;

    /**
     * 每个令牌产生的时间
     */
    private long intervalPerPermit;

    public TokenBucketLimiterPolicy(long bucketMaxTokens, long maxBurstTime) {
        // 最大令牌数
        this.bucketMaxTokens = bucketMaxTokens;
        // 限流时间间隔
        this.resetBucketInterval = 1000;
        if (bucketMaxTokens >= 200) {
            intervalPerPermit = 50;
            initTokens = bucketMaxTokens / 50;
            return;
        }
        // 令牌的产生间隔 = 限流时间 / 最大令牌数
        intervalPerPermit = resetBucketInterval / bucketMaxTokens;
        // 初始令牌数 = 最大的突发流量的持续时间 / 令牌产生间隔
        // 用 最大的突发流量的持续时间 计算的结果更加合理,并不是每次初始化都要将桶装满
        initTokens = Math.min(Math.max(maxBurstTime * bucketMaxTokens / resetBucketInterval, 1), bucketMaxTokens);
    }

    public String[] toParams() {
        List<String > list = Lists.newArrayList();
        list.add(String.valueOf(getIntervalPerPermit()));
        list.add(String.valueOf(System.currentTimeMillis()));
        list.add(String.valueOf(getInitTokens()));
        list.add(String.valueOf(getBucketMaxTokens()));
        list.add(String.valueOf(getResetBucketInterval()));
        return list.toArray(new String[]{});
    }

}
