package cn.com.duiba.spring.boot.starter.dsp.util;

import cn.com.duiba.wolf.utils.NumberUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.scripting.support.StaticScriptSource;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author wangwei
 * @since 2021/12/7 11:10 上午
 */
public class RedisBloomHandler<K, V> {

    private static final String RESERVE = "BF.RESERVE";
    private static final String ADD = "BF.ADD";
    private static final String MADD = "BF.MADD";
    private static final String EXISTS = "BF.EXISTS";
    private static final String MEXISTS = "BF.MEXISTS";

    @Resource(name = "redisBloomStringRedisTemplate")
    private RedisTemplate redisTemplate;

    RedisSerializer keySerializer() {
        return redisTemplate.getKeySerializer();
    }

    RedisSerializer valueSerializer() {
        return redisTemplate.getValueSerializer();
    }

    RedisSerializer hashKeySerializer() {
        return redisTemplate.getHashKeySerializer();
    }

    RedisSerializer hashValueSerializer() {
        return redisTemplate.getHashValueSerializer();
    }

    RedisSerializer stringSerializer() {
        return redisTemplate.getStringSerializer();
    }

    public void createFilter(K key, double errorRate, long initCapacity) {
        byte[] rawKey = rawKey(key);
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            return null;
        }, true);
    }

    public void createFilters(List<K> keys, double errorRate, long initCapacity) {
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            keys.forEach(key -> {
                byte[] rawKey = rawKey(key);
                connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            });
            return null;
        }, true);
    }

    public void createFilterWithExpire(K key, double errorRate, long initCapacity, long seconds) {
        byte[] rawKey = rawKey(key);
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
            connection.expire(rawKey, seconds);
            return null;
        }, true);
    }

    public void createFiltersWithExpire(List<K> keys, double errorRate, long initCapacity, long seconds) {
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            keys.forEach(key -> {
                byte[] rawKey = rawKey(key);
                connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
                connection.expire(rawKey, seconds);
            });
            return null;
        }, true);
    }

    public Boolean add(K key, V value) {
        byte[] rawKey = rawKey(key);
        byte[] rawValue = rawValue(value);
        return (Boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(ADD, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean addString(String key, String value) {
        byte[] rawKey = rawString(key);
        byte[] rawValue = rawString(value);
        return (Boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(ADD, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean addString(RedisTemplate redisTemplate, String key, String value) {
        byte[] rawKey = rawString(key);
        byte[] rawValue = rawString(value);
        return (Boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(ADD, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean[] addMulti(K key, V... values) {
        byte[][] rawArgs = rawArgs(key, values);
        return (Boolean[]) redisTemplate.execute(connection -> {
            List<Long> ls = (List<Long>) connection.execute(MADD, rawArgs);
            return ls.stream().map(l -> Objects.equals(l, 1L)).toArray(Boolean[]::new);
        }, true);
    }

    public boolean exists(K key, V value) {
        byte[] rawKey = rawKey(key);
        byte[] rawValue = rawValue(value);
        return (boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public boolean exists(RedisTemplate redisTemplate, K key, V value) {
        byte[] rawKey = rawKey(key);
        byte[] rawValue = rawValue(value);
        return (boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public boolean existsString(String key, String value) {
        byte[] rawKey = rawString(key);
        byte[] rawValue = rawString(value);
        return (boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public boolean existsString(RedisTemplate redisTemplate, String key, String value) {
        byte[] rawKey = rawString(key);
        byte[] rawValue = rawString(value);
        return (boolean) redisTemplate.execute(connection -> {
            Long l = (Long) connection.execute(EXISTS, rawKey, rawValue);
            return Objects.equals(l, 1L);
        }, true);
    }

    public Boolean[] existsMulti(K key, V... values) {
        byte[][] rawArgs = rawArgs(key, values);
        return (Boolean[]) redisTemplate.execute(connection -> {
            List<Long> ls = (List<Long>) connection.execute(MEXISTS, rawArgs);
            return ls.stream().map(l -> Objects.equals(l, 1L)).toArray(Boolean[]::new);
        }, true);
    }

    public Boolean delete(K key) {
        return redisTemplate.delete(key);
    }

    public Boolean expire(K key, long timeOut, TimeUnit timeUnit) {
        return redisTemplate.expire(key, timeOut, timeUnit);
    }

    public Boolean hasBloom(K key) {
        return redisTemplate.hasKey(key);
    }

    public Boolean hasBloom(RedisTemplate redisTemplate, K key) {
        return redisTemplate.hasKey(key);
    }

    byte[] rawKey(Object key) {
        Assert.notNull(key, "non null key required");
        return this.keySerializer() == null && key instanceof byte[] ? (byte[]) ((byte[]) key) : this.keySerializer().serialize(key);
    }

    byte[] rawString(String key) {
        return this.stringSerializer().serialize(key);
    }

    byte[] rawValue(Object value) {
        return this.valueSerializer() == null && value instanceof byte[] ? (byte[]) ((byte[]) value) : this.valueSerializer().serialize(value);
    }

    private byte[][] rawArgs(Object key, Object... values) {
        byte[][] rawArgs = new byte[1 + values.length][];

        int i = 0;
        rawArgs[i++] = rawKey(key);

        for (Object value : values) {
            rawArgs[i++] = rawValue(value);
        }

        return rawArgs;
    }

    /**
     * 保存数据到布隆过滤器
     *
     * @param key        redis key
     * @param value      值
     * @param levelCount 过滤总层次
     * @return
     */
    public Integer saveBloom(String key, String value, Integer levelCount) {
        String luaText = "local a = "+(levelCount+1)+"\n" +
                "for i = 1, #(ARGV)+1 do\n" +
                "    if(redis.call('BF.ADD', KEYS[1], (ARGV[1] .. '_' .. i)) == 1) then\n" +
                "        a = i;\n" +
                "        break;\n" +
                "    end\n" +
                "end\n" +
                "return a;";
        DefaultRedisScript<Long> redisScript = new DefaultRedisScript<>();
        redisScript.setScriptSource(new StaticScriptSource(luaText));
        redisScript.setResultType(Long.class);
        Object execute = redisTemplate.execute(redisScript, Collections.singletonList(key), value, String.valueOf(levelCount));
        if (NumberUtils.isNumeric(execute + "")) {
            return Integer.valueOf(execute + "");
        }
        return null;
    }

    /**
     * 判断创建redis 布隆过滤器
     *
     * @param keys
     * @param errorRate
     * @param initCapacity
     * @param expireAt
     */
    public void createFiltersWithExpireAt(List<K> keys, double errorRate, long initCapacity, long expireAt) {
        byte[] rawErrorRate = rawString(String.valueOf(errorRate));
        byte[] rawInitCapacity = rawString(String.valueOf(initCapacity));
        Object execute = redisTemplate.execute(connection -> {
            keys.forEach(key -> {
                byte[] rawKey = rawKey(key);
                connection.execute(RESERVE, rawKey, rawErrorRate, rawInitCapacity);
                connection.expireAt(rawKey, expireAt);
            });
            return null;
        }, true);
    }

    /**
     * 根据key判断redis key是否设置过期时间
     *
     * @param key key
     * @return
     */
    public Long getExpire(String key) {
        return redisTemplate.getExpire(key);
    }

    /**
     * 设置redis过去时间
     *
     * @param key   key
     * @param delta 过期时间点
     * @return
     */
    public Boolean expireAt(String key, Date delta) {
        return redisTemplate.expireAt(key, delta);
    }
}
