package cn.com.duiba.spring.boot.starter.dsp.model.service.impl;

import cn.com.duiba.nezha.alg.model.tf.LocalTFModel;
import cn.com.duiba.spring.boot.starter.dsp.model.enums.AlgoTFModelStatusEnum;
import cn.com.duiba.spring.boot.starter.dsp.model.model.AlgoTFModel;
import cn.com.duiba.spring.boot.starter.dsp.model.service.AlgoTFModelProxy;
import cn.com.duiba.spring.boot.starter.dsp.model.support.ChooseTFModelStrategy;
import cn.com.duiba.spring.boot.starter.dsp.model.support.StrategyFactory;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import java.util.Random;

@Slf4j
@Data
public class AlgoTFModelProxyImpl implements AlgoTFModelProxy {

    private static Random random = new Random();

    // 模型key
    private String tfKey;
    // 模型1
    private AlgoTFModel algoTFModel1;
    // 模型2
    private AlgoTFModel algoTFModel2;
    // 模型更新时间
    private long tfModelUpdateTime;

    AlgoTFModelProxyImpl(String tfKey) throws Exception {
        this.tfKey = tfKey;
        LocalTFModel localTFModel = new LocalTFModel();
        try {
            localTFModel.loadModel(tfKey);
        } catch (Exception e) {
            log.warn("localTFModel loadModel error...", e);
            throw e;
        }
        // 初始化投入生产的模型
        algoTFModel1 = new AlgoTFModel(localTFModel, AlgoTFModelStatusEnum.RUNNING.getCode(), tfKey + "-A");
        // 初始化准备投入生产的模型
        algoTFModel2 = new AlgoTFModel(AlgoTFModelStatusEnum.STOP.getCode(), tfKey + "-B");
    }

    @Override
    public LocalTFModel chooseTFModel() {
        // case1
        if (!algoTFModel1.isRunning() && !algoTFModel2.isRunning()) {
            return null;
        }

        // case2
        if (algoTFModel1.isRunning() && !algoTFModel2.isRunning()) {
            return algoTFModel1.getLocalTFModel();
        }

        // case3
        if (!algoTFModel1.isRunning() && algoTFModel2.isRunning()) {
            return algoTFModel2.getLocalTFModel();
        }

        AlgoTFModel newTFModel;
        AlgoTFModel oldTFModel;
        if (algoTFModel1.getLoadTime().compareTo(algoTFModel2.getLoadTime()) > 0) {
            newTFModel = algoTFModel1;
            oldTFModel = algoTFModel2;
        } else {
            newTFModel = algoTFModel2;
            oldTFModel = algoTFModel1;
        }
        Long loadTime = newTFModel.getLoadTime();

        if (System.currentTimeMillis() - loadTime >= 90000) {
            return newTFModel.getLocalTFModel();
        }

        ChooseTFModelStrategy strategy = StrategyFactory.getStrategy();
        if (strategy.tryAcquire(loadTime)) {
            return newTFModel.getLocalTFModel();
        }
        return oldTFModel.getLocalTFModel();

    }

    @Override
    public void closeTFModel() {
        boolean allowClose1 = algoTFModel1.isAllowClose();
        boolean allowClose2 = algoTFModel2.isAllowClose();

        // 如果模型1超过5分钟没有被访问，进行关闭
        if (allowClose1) {
            algoTFModel1.doCloseTFModel();
        }

        // 如果模型2超过5分钟没有被访问，进行关闭
        if (allowClose2) {
            algoTFModel2.doCloseTFModel();
        }

    }

    @Override
    public boolean updateTFModel() {
        // case1：模型1停止运行，并且模型2停止运行
        // 则代表此key对应的模型已经停止使用了, 不做处理，继续下一个循环逻辑
        if (!algoTFModel1.isRunning() && !algoTFModel2.isRunning()) {
            return false;
        }
        // case2：模型1正在运行，模型2停止运行
        // 则立刻加载新模型到模型2上
        LocalTFModel localTFModel = new LocalTFModel();
        if (algoTFModel1.isRunning() && !algoTFModel2.isRunning() && algoTFModel1.isLatestVersion(localTFModel, tfKey)) {
            algoTFModel2.loadTFModel(localTFModel, tfKey);
            tfModelUpdateTime = System.currentTimeMillis();
            return true;
        }

        // case3：模型1停止运行，模型2正在运行
        // 则立刻加载新模型到模型1上
        if (!algoTFModel1.isRunning() && algoTFModel2.isRunning() && algoTFModel2.isLatestVersion(localTFModel, tfKey)) {
            algoTFModel1.loadTFModel(localTFModel, tfKey);
            tfModelUpdateTime = System.currentTimeMillis();
            return true;
        }
        return false;
    }

    @Override
    public boolean hasTwoRunningModel() {
        return algoTFModel1.isRunning() && algoTFModel2.isRunning();
    }

    @Override
    public boolean needFlush() {
        return !algoTFModel1.isRunning() && !algoTFModel2.isRunning();
    }

}
