package cn.com.duiba.spring.boot.starter.dsp.warmup;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.AopTargetUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.github.benmanes.caffeine.cache.AsyncCache;
import com.google.common.cache.Cache;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

/**
 * @author shenjiaqing
 * @Description
 */
@Configuration
@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE)
public class CacheConfig {
    private static final Logger logger = LoggerFactory.getLogger(CacheConfig.class);

    @Autowired
    private ConfigurableApplicationContext applicationContext;

    @Resource(name = "eurekaClient")
    private EurekaClient eurekaClient;

    @Value("${spring.application.name}")
    private String appName;

    /**
     * 本地缓存预热开关：默认关闭
     */
    @Value("${local.cache.warmup.enable:false}")
    private Boolean cacheEnable;

    @EventListener(MainContextRefreshedEvent.class)
    public void onMainContextRefreshed(){
        String[] names = applicationContext.getBeanDefinitionNames();
        for (String beanName : names) {
            try {
                Object bean = applicationContext.getBean(beanName);
                Class<?> beanObjClass = bean.getClass();
                Package pkg = beanObjClass.getPackage();
                if (pkg == null) {
                    continue;
                }
                String packageName = pkg.getName();
                if (packageName.startsWith("springfox.") || packageName.startsWith("org.")
                        || packageName.startsWith("io.") || packageName.startsWith("net.")
                        || packageName.startsWith("cn.com.duibaboot") || packageName.startsWith("com.netflix")
                        || packageName.startsWith("cn.com.duiba.tuia.pangea")
                        || beanName.equals("loadModelService")) {
                    continue;
                }

                // 判断类是否是代理类
                if (AopUtils.isAopProxy(bean)) {
                    try {
                        bean = AopTargetUtils.getTarget(bean);
                    } catch (Exception e) {
                        logger.error("代理类获取原始类失败");
                    }
                }

                Field[] fields = bean.getClass().getDeclaredFields();
                if (fields.length == 0) {
                    continue;
                }

                for (Field field : fields) {
                    boolean isGuava = Cache.class.isAssignableFrom(field.getType());
                    boolean isCaffeine = AsyncCache.class.isAssignableFrom(field.getType());

                    if (!isCaffeine && !isGuava) {
                        continue;
                    }
                    field.setAccessible(true);
                    String keyName = String.format("%s-%s", beanName, field.getName());
                    try {
                        Object cache = field.get(bean);
                        if (cache instanceof Cache) {
                            ConcurrentMap<Object, Object> concurrentMap = ((Cache) cache).asMap();
                            CacheManager.saveCache(keyName, concurrentMap);
                            logger.info("beanName = {}, cache = {}, isGuava = {}, bean = {}", keyName, cache, isGuava, bean);
                        }

                        if (cache instanceof AsyncCache) {
                            ConcurrentMap<Object, Object> concurrentMap = ((AsyncCache) cache).synchronous().asMap();
                            CacheManager.saveCache(keyName, concurrentMap);
                            logger.info("beanName = {}, cache = {}, isCaffeine = {}, bean = {}", keyName, cache, isCaffeine, bean);
                        }
                    } catch (IllegalAccessException e) {
                        //ignore
                    } finally {
                        field.setAccessible(false);
                    }
                }
            }catch(Exception e){
                logger.warn("cache实例获取异常", e);
            }
        }
    }

    // 顺序最靠后 等其他地方处理MainContextRefreshedEvent事件完成后
    @EventListener(MainContextRefreshedEvent.class)
    @Order(Ordered.LOWEST_PRECEDENCE-1)
    public void onServerCache(){
        try{
            // 本地缓存预热开关 : 默认关闭
            if (!cacheEnable) {
                logger.warn("Cache preheating failed，because cacheEnable is false");
                return;
            }
            List<InstanceInfo> instances = eurekaClient.getApplication(appName).getInstances();
            if (CollectionUtils.isEmpty(instances)) {
                logger.warn("获取容器列表为空!");
                return;
            }
            logger.info("container of all registered AppInfo size: {}", instances.size());
            List<InstanceInfo> list = instances.stream().sorted((o1, o2) -> (int) (o2.getLeaseInfo().getServiceUpTimestamp()-o1.getLeaseInfo().getServiceUpTimestamp())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(list)) {
                logger.warn("Cache preheating failed，because 服务启动获取【" + appName + "】列表信息为空");
                return;
            }
            logger.info("打印warmup获取备份数据实例信息：{}", JSONObject.toJSONString(list.get(0)));
            String result = HttpRequestUtils.sendHttp(list.get(0).getHomePageUrl() + "/local/getCacheMap", null, appName);
            if (StringUtils.isBlank(result)) {
                logger.warn("Cache preheating failed，because request {} ,response {}", list.get(0).getHomePageUrl() + "/local/getCacheMap", result);
                return;
            }

            ObjectMapper objectMapper = new ObjectMapper();
            TypeFactory typeFactory = objectMapper.getTypeFactory();
            Map<String, Object> map = objectMapper.convertValue(JSONObject.parse(result), typeFactory.constructMapType(Map.class, String.class, Object.class));
            map.forEach((k, v) -> {
                // beanName----cacheFieldName
                String[] split = k.split("-");
                Object bean = applicationContext.getBean(split[0]);
                // 判断类是否是代理类
                if (AopUtils.isAopProxy(bean)) {
                    try {
                        bean = AopTargetUtils.getTarget(bean);
                    } catch (Exception e) {
                        logger.error("代理类获取原始类失败");
                    }
                }
                Field[] fields = bean.getClass().getDeclaredFields();
                for (Field field : fields) {
                    if (field.getName().equals(split[1]) && ((JSONObject) JSONObject.parse(JSON.toJSONString(v))).size() > 0) {
                        try {
                            field.setAccessible(true);
                            Object obj = field.get(bean);

                            if (!(obj instanceof Cache) && !(obj instanceof AsyncCache)) {
                                continue;
                            }

                            // 通过MapType解析map结构
                            ParameterizedType genericType = (ParameterizedType) field.getGenericType();
                            Type[] actualTypeArguments = genericType.getActualTypeArguments();
                            MapType mapType = typeFactory.constructMapType(Map.class, (Class<?>) actualTypeArguments[0], Object.class);
                            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                            Map convertMap = objectMapper.convertValue(v, mapType);

                            if (obj instanceof Cache) {
                                Cache cache = (Cache) obj;

                                convertMap.forEach((k1, v1) -> {
                                    // 通过JavaType解析map中的value结构
                                    ParameterizedType genericType1 = (ParameterizedType) actualTypeArguments[1];
                                    Type[] actualTypeArguments1 = genericType1.getActualTypeArguments();
                                    JavaType javaType = typeFactory.constructType(actualTypeArguments1[0]);
                                    cache.put(k1, Optional.ofNullable(objectMapper.convertValue(v1, javaType)));
                                });
                                logger.info("Guava Cache key = {} ; Guava Cache Size = {}", k, cache.asMap().size());
                            }


                            if (obj instanceof AsyncCache) {
                                AsyncCache asyncCache = (AsyncCache) obj;

                                convertMap.forEach((k1, v1) -> {
                                    // 通过JavaType解析map中的value结构
                                    ParameterizedType genericType1 = (ParameterizedType) actualTypeArguments[1];
                                    Type[] actualTypeArguments1 = genericType1.getActualTypeArguments();
                                    JavaType javaType = typeFactory.constructType(actualTypeArguments1[0]);
                                    asyncCache.put(k1, CompletableFuture.supplyAsync(() -> Optional.ofNullable(objectMapper.convertValue(v1, javaType))));
                                });
                                logger.info("Caffeine Cache key = {} ; Caffeine Cache Size = {}", k, asyncCache.synchronous().asMap().size());
                            }

                            field.setAccessible(false);
                        } catch (Exception e) {
                            logger.warn("this bean : {} , no have method : {}, Lead to it cannot be deserialized , cause: {}", split[0], field.getName() + " Serialize", e);
                        }
                    }
                }
            });
        } catch (Exception e) {
            logger.warn("服务启动获取【" + appName +"】列表信息失败", e);
        }

    }




}
