import { 
    promisifyApiSimple, 
    promisifyApi, 
    promisifyApiToTb, 
    AudioModel
} from '../helpers'
/**
 * 通用toast
 * @param {string} content toast内容
 * @param {number} duration 显示时长
 * @param {function} successCb 成功回调
 */
export const commonToast = (content: string, duration?: number, successCb?: () => any) => {
    promisifyApiSimple('showToast',{
        content: content || '换个姿势再试一次~',
        duration: duration || 3000,
        success: successCb
    })
};
/**
 * 简易alert
 * @param {string} content toast内容
 */
export const simpleAlert = (content: string) => {
    promisifyApiSimple('alert',{
        content
    })
};
/**
 * 通用确认弹窗
 * @param {string} title 标题
 * @param {string} content 内容
 * @param {function} successCb 成功回调
 */
export const commonConfirm = async (title = '提示', content: string, successCb?: () => any) => {
    promisifyApiSimple('confirm',{
        title,
        content,
        confirmButtonText: '确定',
        cancelButtonText: '取消',
        success: result => {
            const { confirm } = result;
            if (confirm) {
              successCb && successCb();
            }
        }
    })
};
/**
 * 获取系统信息
 */
export const getSystemInfo = async () => {
    return await promisifyApi('getSystemInfo', {})
};
/**
 * 执行关注店铺
 * @param {number} sellerId 店铺归属的卖家Id
 * @returns {boolean} 关注状态
 */
export const favorShop = async (sellerId: number) => {
    if (!sellerId) {
      return false;
    }
    let result: { error?: number } = await promisifyApiToTb('favorShop',{ id: +sellerId })
    if(result?.error === 11) return false
    return result
};
/**
 * 判断是否关注店铺
 * @param {number} sellerId 店铺归属的卖家Id
 * @returns {boolean} 关注状态
 */
export const checkShopFavoredStatus = async (sellerId: number) => {
    if (!sellerId) {
      return false;
    }
    let result: { isFavor?: boolean } = await promisifyApiToTb('checkShopFavoredStatus',{ id: +sellerId })
    if(result?.isFavor) return result.isFavor
    if(result === 'checkShopFavoredStatus 仅支持真机调用，请在手机淘宝中运行')return true
    return result
};
/**
 * 跳转到外部链接
 * @param {string} url 跳转链接
 */
export const navigateToOutside = url => {
    if (!url) {
      console.error('navigateToOutside: 请传入url');
      return false;
    }
    promisifyApiSimple('call',{
        url
    }, 'navigateToOutside')
};
/**
 * 跳转到内部链接（新开窗口）
 * @param {string} url 跳转链接
 */
export const navigateTo = url => {
    if (!url) {
      console.error('navigateTo: 请传入url');
      return false;
    }
    promisifyApiSimple('navigateTo',{
        url
    })
};
/**
 * 返回页面
 * @param {number} delta
 */
export const navigateBack = (delta?: number) => {
    promisifyApiSimple('navigateBack',{
        delta
    })
};
/**
 * 跳转到内部链接（不新开窗口）
 * @param {string} url
 */
export const redirectTo = (url: string) => {
    if (!url) {
      console.error('redirectTo: 请传入url');
      return false;
    }
    promisifyApiSimple('redirectTo',{
        url
    })
};
/**
 * 获取服务器时间
 * @returns {number} 服务器时间戳
 */
export const getServerTime = async () => {
    const result: { time?: string } = await promisifyApi('getServerTime', {})
    if(result?.time) return result.time
    return result
};
/**
 * 收藏商品
 * @param {number} 商品id
 * @returns {object} 收藏结果
 */
export const collectGoods = async (id: number) => {
    if (!id) {
      console.error('collectGoods: 请传入商品id');
      return false;
    }
    let result: { error?: number } = await promisifyApiToTb('collectGoods',{ id: +id })
    if(result?.error === 11) return false
    return result
};
/**
 * 查询商品收藏状态
 * @param {number} 商品id
 * @returns {boolean} 商品收场状态
 */
export const checkGoodsCollectedStatus = async (id: number) => {
    if (!id) {
      console.error('checkGoodsCollectedStatus: 请传入商品id');
      return false;
    }
    let result: { isCollect?: boolean } = await promisifyApiToTb('checkGoodsCollectedStatus',{ id: +id })
    if(result?.isCollect) return result.isCollect
    return result
};
/**
 * 跳转到淘宝商品页
 * @param {string} itemId 商品ID
 */
export const openDetail = async (itemId: string) => {
    if (!itemId) {
      console.error('openDetail: 请传入商品id');
      return false;
    }
    return await promisifyApiToTb('openDetail',{ itemId: String(itemId) })
};
/**
 * 创建音频实例
 * @param {object} opts 配置对象 例 { src: '音频链接', autoplay: false }
 */
export const createAudio = (opts: { src: string, autoplay?: boolean}) => {
    const audioInstance = new AudioModel(opts);
    return audioInstance;
};
/**
 * 获取图片信息
 * @param {string} src 图片地址
 */
export const getImageInfo = async (src: string) => {
    return await promisifyApi('getImageInfo', { src })
};
/**
 * 剪切板
 * @param {text: string} 
 */
export const setClipboard = async (text: string) => {
	return await promisifyApi('setClipboard', { text })
};
/**
 * 调起分享面板
 */
export const showSharePanel = async () => {
    promisifyApiSimple('showSharePanel',{})
}
/**
 * 获取缓存数据
 * @param key 
 */
export const getStorage = async (key: string) => {
    const result: { data?: string} = await promisifyApi('getStorage', { key })
    if(result?.data) return result.data
    return result
}
/**
 * 设置缓存数据
 * @param key 
 * @param value 
 */
export const setStorage = async (key: string, value: any) => {
    return await promisifyApi('setStorage', { key, data: value })
}
/**
 * 清除缓存数据
 * @param key 
 */
export const removeStorage = async (key: string) => {
    return await promisifyApi('removeStorage', { key })
}
/**
 * 自定义分析数据的上报接口
 * @param eventType 
 * @param logkey 
 */
export const taobaoReportClick = (eventType: string,logkey: string) => {
    // @ts-ignore
    my.reportAnalytics(eventType, {
        // 标志某一类事件，方便数据分析进行分组
        'logkey': logkey
    })
};
/**
 * 打开BC客服聊天
 * @param sellerNick 
 * @param itemId 
 */
export const openMessage = async (sellerNick: string, itemId: string) => {
    let result = await promisifyApiToTb('openMessage',{ sellerNick, forceH5: false, params: { itemId } })
    return result
}
/**
 * 显示指定商品 SKU 选择器
 * @param itemId 
 */
export const showSkuModal = async (itemId: string) => {
    let result = await promisifyApiToTb('showSku',{ itemId })
    return result
}
/**
 * 隐藏正在展示的商品 SKU 选择器
 */
export const hideSkuModal = async () => {
    let result = await promisifyApiToTb('hideSku',{})
    return result
}
/**
 * 跳转订单页
 * @param itemId 
 * @param skuId 
 * @param quantity 
 */
export const confirmCustomOrder = async (itemId: number, skuId?: number, quantity?: number) => {
    let result = await promisifyApiToTb('confirmCustomOrder',{ 
        data: { 
            itemId, 
            skuId: skuId || -1, 
            quantity: quantity || 1, 
            customization: {} 
        }
    })
    return result
}