import { commonToast, getServerTime } from '../core/tb';
/**
 * 时间格式化
 * @param date    接收可以被new Date()方法转换的内容
 * @param format  字符串，需要的格式例如：'yyyy/MM/dd hh:mm:ss'
 * @returns {String}
 */
export const dateFormatter = (dateTxt: string, format = 'yyyy/MM/dd') => {
    if (!dateTxt) return '-';
  
    const date = new Date(
      typeof dateTxt === 'string' && isNaN(+dateTxt) ? dateTxt.replace(/-/g, '/') : dateTxt
    );
    let o = {
      'M+': date.getMonth() + 1,
      'd+': date.getDate(),
      'h+': date.getHours(),
      'm+': date.getMinutes(),
      's+': date.getSeconds(),
      'q+': Math.floor((date.getMonth() + 3) / 3),
      S: date.getMilliseconds()
    };
    if (/(y+)/.test(format)) {
      format = format.replace(
        RegExp.$1,
        (date.getFullYear() + '').substr(4 - RegExp.$1.length)
      );
    }
    for (let k in o) {
      if (new RegExp('(' + k + ')').test(format)) {
        format = format.replace(
          RegExp.$1,
          RegExp.$1.length === 1 ? o[k] : ('00' + o[k]).substr(('' + o[k]).length)
        );
      }
    }
    return format;
};
/**
 * 版本比较
 * @param {string} cur 当前版本
 * @param {string} compare 比较版本
 */
export const versionCompare = (cur: string, compare: string) => {
    const _cur = cur.split('.');
    const _compare = compare.split('.');
    if (+_cur[0] < +_compare[0]) {
      return false;
    } else if ( +_cur[0] === +_compare[0] && +_cur[1] < +_compare[1] ) {
      return false;
    } else if (+_cur[0] === +_compare[0] && +_cur[1] === +_compare[1] && +_cur[2] < +_compare[2]) {
      return false;
    }
    return true;
};
/**
 * 获取request请求所需参数
 * @param {object/string} value API项配置
 */
export const getRequestParams = (value: any) => {
    if (typeof value === 'string') {
      return {
        handle: value,
        method: 'POST',
        ext: {}
      }
    } else if (typeof value === 'object') {
      const { handle, method = 'POST', defaultParams = {}, ...ext } = value;
      return {
        handle,
        method,
        defaultParams,
        ext: { ...ext }
      }
    } else {
      console.error('getRequestParams: 传参有误');
    }
};

type IApis<T> = {
  [P in keyof T]: (params?: object, config?: {
    isShowLoading?: boolean,  // 请求是否loading
    toastError?: boolean // 请求失败是否 toast错误
 }) => {}
}

/**
 * 生成API
 * @param {array} apiList API数据数组
 * @param {function} request 请求
 */
export const generateAPI = <T, K extends Function>(apiList: T, request: K): IApis<T> => {
  // @ts-ignore
    const api: IApis<typeof apiList> = {};
    Object.entries(apiList).forEach(([key, value]) => {
      const { handle, method, ext, defaultParams } = getRequestParams(value);
      api[key] = (params, config = {}) => request(handle, method,  Object.assign({}, defaultParams, params) ,  Object.assign({}, ext, config));
    });
  
  return api;
  // return Object.keys(apiList).reduce((prev, curr) => {
  //     const { handle, method, ext, defaultParams } = getRequestParams(apiList[curr]);
  //   return {
  //     ...prev,
  //     [curr]: params => request(handle, method,  Object.assign({}, defaultParams, params) , ext)
  //     }
  //   }, {})
};
/**
 * 函数节流，普通防连点
 * @param {fun} fun 函数
 * @param {delay} delay 时间
 */
export const throttleHandle = (fun: (t: any) => any, delay: number = 1000) => {
    let last: number, deferTimer: any;
    return function () {
      let now = +new Date();
      if (last && now < last + delay) {
        clearTimeout(deferTimer);
        deferTimer = setTimeout(() => {
          last = now;
        }, delay);
      } else {
        last = now;
        fun.apply(this, arguments);
      }
    };
}
/**
 * 根据fileId获取图片地址
 * @param fileId 
 */
export const getImgShareUrl = async (fileId: string | Array<string>) => {
    // @ts-ignore
    const { cloud } = getApp()
    let list = await cloud.file.getTempFileURL({ fileId })
    if(typeof fileId === 'string') return list && list[0].url
    return list
}
/**
 * 根据sellerId判断入会状态
 * @param sellerId 
 */
export const checkIsMember = async (sellerId: number) => {
  // @ts-ignore
  const  memberPlugin = requirePlugin("cemMember");
  return new Promise((resolve,reject) => {
    memberPlugin.checkMember({
      sellerId,
      success: (res) => {
        console.log('会员检查成功 :>> ', res)
        let isMember = res?.businessSuccess && res?.data?.isMember === 'true'
        resolve(isMember)
      },
      fail: (res) => {
        console.log('会员检查失败 :>> ', res)
        console.warn('checkMember 需在真机上调试')
        reject(res)
      }
    })
  })
}
/**
 * 毫秒转换成00:00:00格式
 * @param ms 毫秒
 */
export const formatTimeStamp = (ms: number) => {
  let hours:  number | string = parseInt(String(ms / (1000 * 60 * 60))),
    minutes:  number | string = parseInt(String((ms % (1000 * 60 * 60)) / (1000 * 60))),
    seconds:  number | string = parseInt(String((ms % (1000 * 60)) / 1000));
    hours = hours < 10 ? '0' + hours : hours;
    minutes = minutes < 10 ? '0' + minutes : minutes;
    seconds = seconds < 10 ? '0' + seconds : seconds;
    return `${hours}:${minutes}:${seconds}`
}
/**
 * 活动状态验证
 * @param startTime 
 * @param endTime 
 * @param noCheck 
 */
export const validateActivityStatus = async (startTime: number, endTime: number, noCheck: boolean) => {
  const nowTime = await getServerTime()
  return new Promise((resolve,reject) => {
    if(nowTime < startTime ) {
      commonToast('活动未开始')
      resolve(false)
    }
    else if(nowTime > endTime && !noCheck) {
      commonToast('活动已结束')
      resolve(false)
    }
    else {
      resolve(true)
    }
  })
}