/**
 * 基本信息
 *
 * @format
 */

import { BaseDao, TBAPIS } from '../../sdk'
import { JOIN_DB_NAME } from '../../db'
import { ACTIVITY_STATUS } from '../../constants'
import { getToday } from '../../utils'
import UserService from './user.service'

export default class GameService extends UserService {
  context: IContext<IParams>
  joindao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.joindao = new BaseDao(context, JOIN_DB_NAME)
  }

  async addJoinRecord(userInfo: IUserInfo) {
    const {
      openId,
      data: { activityId }
    } = this.context
    const { userNick, avatar } = userInfo
    const now = Date.now()
    const today = getToday()
    const record: IJoinRecord = {
      activityId,
      userNick,
      avatar,
      openId,
      createTime: now,
      updateTime: now,
      score: 0,
      createDay: today
    }

    const id = await this.joindao.insertOne(record)

    return { id }
  }

  async submitGame(
    id: string,
    score: number,
    rankKey: string,
    rankTime: string,
    userInfo: IUserInfo,
    joinInfo: IJoinRecord
  ) {
    const { duration } = await this.updateJoinRecord(id, score, userInfo, joinInfo)

    const { maxScore, totalScore } = await this.updateUserScore(score, userInfo)

    const myRankInfo = await this.getMyRankInfo(rankKey, rankTime, userInfo)

    const { gameTimes, avatar, userNick } = userInfo

    return {
      score,
      maxScore,
      totalScore,
      gameTimes,
      duration,
      avatar,
      userNick,
      rank: myRankInfo.rank
    }
  }

  async updateJoinRecord(id: string, score: number, userInfo: IUserInfo, joinInfo: IJoinRecord) {
    const now = Date.now()
    const today = getToday()
    const duration = now - joinInfo.createTime
    const playInfo = {
      score,
      duration,
      submitTime: now,
      updateTime: now,
      submitDay: today
    }

    await this.joindao.update({ _id: id }, { $set: playInfo })

    return playInfo
  }

  async updateUserScore(score: number, userInfo: IUserInfo) {
    const now = Date.now()
    const newMaxScore = score >= userInfo?.maxScore
    const maxScore = newMaxScore ? score : userInfo?.maxScore
    const updateScoreTime = newMaxScore ? now : userInfo?.updateScoreTime

    const { _id, totalScore } = userInfo

    await this.updateUser(_id, {
      $set: { maxScore, updateScoreTime },
      $inc: { totalScore: score }
    })

    return {
      maxScore,
      updateScoreTime,
      totalScore: totalScore + score
    }
  }
}
