/**
 * 任务相关方法
 *
 * @format
 */

import UserService from './user.service'
import { getToday, getUserOrderlist, generateVipUrl, formatVipCbUrl } from '../../utils'
import { TASK_RATE_TYPE, TASK_STATUS } from '../../constants'
import { getTodayCompleteTask, getTotalCompleteTask, setTaskStatus } from '../../utils/common/task'
import { CODE_TYPES } from '../../errorCode'

export interface ITaskInfo {
  taskType?: string // 任务类型
  url?: string // 链接地址
  rewards: number // 任务奖励
  itemIds?: string // 商品配置ids 商品相关任务返回
  todayCompleteTimes?: number // 今日完成次数
  completeTimes?: number // 总共完成次数
  taskRateType: number // 任务频率
  times?: number // 任务每日限制次数
  waitReceive?: number // 待领取次数
  title: string // 任务文案
  status: number // 任务状态 1未完成 2 待领取 3 已完成
}
export default class TaskService extends UserService {
  constructor(context: IContext<IParams>) {
    super(context)
  }

  // 根据活动tasks字段渲染任务
  async initTaskList(userInfo: IUserInfo, activityInfo: IActivityInfo) {
    return {
      list: Object.keys(activityInfo?.tasks || {}).map((task: ITaskType) => {
        return this.initTask(task, activityInfo?.tasks?.[task]?.title, userInfo, activityInfo)
      })
    }
  }
  /**
   *  根据任务类型初始化   b端 tasks字段必须有相应的配置
   *
   * @param {ITaskType} taskType 任务类型
   * @param {string} title 任务标题
   * @param {IActivityInfo} activityInfo 任务配置
   * @param {IUserInfo} userInfo 用户信息
   * @return {ITaskInfo}  {ITaskInfo} 任务信息
   * @memberof TaskService
   */
  initTask(taskType: ITaskType, title: string, userInfo: IUserInfo, activityInfo: IActivityInfo): ITaskInfo {
    if (!activityInfo?.tasks?.[taskType]) {
      console.error(`活动缺少${taskType}任务配置项`)
    }
    const { value, itemIds = '', taskRateType, times = 1, link = '' } = activityInfo?.tasks?.[taskType] || {}
    const { remainTimes } = userInfo
    const { todayCompleteTimes } = getTodayCompleteTask(taskType, userInfo)
    const { completeTimes } = getTotalCompleteTask(taskType, userInfo)

    return {
      taskType,
      title,
      itemIds,
      rewards: value,
      taskRateType,
      times,
      url: taskType === 'member' ? generateVipUrl(formatVipCbUrl(this.context)) : link,
      todayCompleteTimes,
      completeTimes,
      status: setTaskStatus(userInfo, taskType, taskRateType, times),
      waitReceive: remainTimes?.[taskType]
    }
  }

  /**
   *
   * 领取任务奖励
   *
   * @param {ITaskType} taskType
   * @param {string} awardsTargetKey
   * @param {IUserInfo} userInfo
   * @return {number} rewards
   * @memberof TaskService
   */
  async receiveTaskRewards(
    taskType: ITaskType,
    awardsTargetKey: string,
    userInfo: IUserInfo
  ): Promise<{
    rewards: number
  }> {
    const waitReceiveTimes = userInfo?.remainTimes?.[taskType]

    await this.updateUser(userInfo._id, {
      $set: {
        [`remainTimes.${taskType}`]: 0
      },
      $inc: {
        [awardsTargetKey]: waitReceiveTimes
      }
    })

    return {
      rewards: waitReceiveTimes
    }
  }

  /**
   *根据任务类型更新任务待领取次数和任务记录
   *
   * @param {string} taskType
   * @param {number} activityInfo
   * @param {IActivityInfo} userInfo
   * @returns {boolean}
   * @memberof TaskService
   */
  async completeTask(taskType: ITaskType, activityInfo: IActivityInfo, userInfo: IUserInfo, customRecord: Object = {}) {
    const today = getToday()
    const rewards = activityInfo?.tasks?.[taskType]?.value || 0
    const {
      openId,
      data: { itemId }
    } = this.context

    const record = { itemId, openId: taskType === 'invites' ? openId : undefined, ...customRecord }
    const result = await this.updateUser(userInfo._id, {
      $inc: {
        [`remainTimes.${taskType}`]: rewards
      },
      $push: {
        [`taskInfo.${today}.${taskType}`]: { ...record, createTime: Date.now() }
      }
    })
    return result ? { ok: 1 } : CODE_TYPES.ERROR_DO_TASK
  }

  // 根据下单下单记录,更新状态
  async updateOrderGoodsTask(userInfo: IUserInfo, tasks: ITasks, activityStartTime: number, session?: string) {
    const taskType = 'orderGoods'
    const {
      [taskType]: { value, itemIds, taskRateType, times }
    } = tasks
    const { completeTimes, taskInfo } = getTotalCompleteTask(taskType, userInfo)
    const { todayCompleteTimes } = getTodayCompleteTask(taskType, userInfo)
    // 永久任务且已完成
    if (taskRateType === TASK_RATE_TYPE.FOREVER && completeTimes) {
      return {}
    }

    // 每日限制完成且完成次数达到限制
    const today = getToday()
    if (taskRateType === TASK_RATE_TYPE.EVERYDAY && todayCompleteTimes >= times) {
      return {}
    }

    const orderResult = await getUserOrderlist(
      this.context,
      //@ts-ignore
      activityStartTime || Date.now(),
      Date.now(),
      session
    )

    const itemIdsArr = itemIds.split(',').map(v => +v)

    let projection = {
      $inc: {
        [`remainTimes.${taskType}`]: 0
      },
      $set: {}
    }

    let targetOrders = userInfo?.taskInfo?.[today]?.[taskType] || []
    orderResult.forEach(v => {
      // @ts-ignore
      // 商品订单包含目标商品 且orderId为新订单
      if (itemIdsArr.includes(v.itemId) && !taskInfo.some(order => order.orderId === v.orderId)) {
        if (taskRateType === TASK_RATE_TYPE.EVERYDAY && targetOrders.length >= times) {
          return
        }
        projection.$inc[`remainTimes.${taskType}`] += +value
        targetOrders.push({
          itemId: v.itemId,
          orderId: v.orderId,
          payTime: v.payTime,
          createTime: Date.now()
        })
      }
      if (targetOrders?.length) {
        projection.$set[`taskInfo.${today}.${taskType}`] = targetOrders
      }
    })

    return projection
  }

  async getItemListWithCollectStatus(list: { list: ITaoBaoItems[] }, userInfo: IUserInfo) {
    const { taskInfo } = getTotalCompleteTask('collectGoods', userInfo)

    return list.list.map(v => {
      return {
        ...v,
        // 完成列表是否含有itemId
        collected: taskInfo.some(completeItem => completeItem.itemId === v.itemId)
      }
    })
  }
}
