/**
 * 基本信息
 *
 * @format
 */

import { BaseDao, resultsModel, TBAPIS } from '../../sdk'
import { JOIN_DB_NAME } from '../../db'
import { ACTIVITY_STATUS, CARD_PRIZE_STATUS, PRIZE_DATA_TYPE, PRIZE_TYPE } from '../../constants'
import { getToday } from '../../utils'
import UserService from './user.service'
import AwardService from './awards.service'
import { CODE_TYPES } from '../../errorCode'
import { uniq } from 'lodash'

export default class CardService extends UserService {
  context: IContext<IParams>
  joindao: IBaseDao
  awardService: AwardService
  constructor(context: IContext<IParams>) {
    super(context)
    this.joindao = new BaseDao(context, JOIN_DB_NAME)
    this.awardService = new AwardService(context)
  }

  // 集卡
  async collectCard(activityId: string, userInfo: IUserInfo) {
    const card = await this.getCard(activityId)

    const { cardType, _id, image, name, type } = card

    if (type === PRIZE_TYPE.THANKS)
      return {
        type: PRIZE_TYPE.THANKS,
        name: '谢谢参与'
      }

    const reduceResult = await this.awardService.reduceStock(_id)
    if ((reduceResult as ICodeType)?.code) return CODE_TYPES.ERROR_NO_STOCK

    userInfo.cardInfo[cardType] = (userInfo.cardInfo?.[cardType] || 0) + 1

    const updateResult = await this.updateUser(userInfo._id, {
      $inc: {
        joinedTimes: 1,
        [`cardInfo.${cardType}`]: 1
      }
    })
    if (updateResult !== 1) return CODE_TYPES.SYSTEM_ERROR

    const myCardInfo = this.getMyCardInfo(userInfo, cardType)

    return {
      cardType,
      image,
      name,
      type,
      ...myCardInfo
    }
  }

  getMyCardInfo(userInfo: IUserInfo, cardType?: number) {
    const { cardInfo } = userInfo
    // 该卡片收集数量
    const currCardCollectedCount = cardType ? cardInfo?.[cardType] || 0 : undefined

    // 已收集卡片的总数量
    const cardQuantityCollectedCount = Object.values(cardInfo).reduce((total, count) => total + count, 0)

    // 已收集卡片类型
    const cardTypeCollectedCount = Object.keys(cardInfo).length
    return {
      currCardCollectedCount,
      cardQuantityCollectedCount,
      cardTypeCollectedCount
    }
  }

  async addCollectRecord(userInfo: IUserInfo, cardResult: object) {
    const { openId } = this.context
    const { activityId } = this.context.data
    const { userNick } = userInfo
    const recordId = await this.joindao.insertOne({
      activityId,
      userNick,
      openId,
      ...cardResult,
      createTime: Date.now(),
      createDay: getToday()
    })
    return { recordId }
  }

  // 获取卡片
  async getCard(activityId: string) {
    const cardPool = await this.awardService.getPrizeConfig({
      activityId,
      prizeDataType: PRIZE_DATA_TYPE.CARD
    })

    let card = await this.awardService.getPrizeByProbability(cardPool)

    const thanksPrize = cardPool.find(v => v.type === PRIZE_TYPE.THANKS) || {
      type: PRIZE_TYPE.THANKS,
      prizeDataType: PRIZE_DATA_TYPE.CARD,
      name: '谢谢参与'
    }

    !card && (card = thanksPrize)

    return card
  }

  async getCardPrizeList(activityId: string, prizeDataType: number, needKey: string, userCount: number) {
    const { openId } = this.context

    const joinedTimesPrizePool = await this.awardService.getPrizeConfig(
      {
        activityId,
        prizeDataType
      },
      {
        projection: { type: 1, image: 1, prizeDataType: 1, name: 1 }
      }
    )

    // 抽奖次数奖池分类 从小到大排列
    const category = uniq(joinedTimesPrizePool.map(v => +v[needKey])).sort()

    const prizeMapper = category.reduce((prev, curr) => {
      return {
        ...prev,
        [`${curr}`]: joinedTimesPrizePool.filter(v => v[needKey] === curr)
      }
    }, {})

    const awardList = await this.awardService.getAwardsInfoList(
      {
        activityId,
        openId,
        prizeDataType
      },
      {
        projection: { type: 1, image: 1, prizeDataType: 1, name: 1 }
      }
    )

    const prizeList = category.map(key => {
      return {
        [needKey]: key,
        status: setCardPrizeStatus(userCount, key, awardList),
        prizeList: prizeMapper[key]
      }
    })

    function setCardPrizeStatus(userCount: number, needCount: number, awardList: IAwards[]) {
      if (userCount < needCount) return CARD_PRIZE_STATUS.LOCK

      if (needCount >= userCount && !awardList.some(v => v[needKey] === needCount)) return CARD_PRIZE_STATUS.UN_LOCK

      if (needCount >= userCount && awardList.some(v => v[needKey] === needCount)) return CARD_PRIZE_STATUS.SUCCESS
    }

    return prizeList
  }

  async getAwardsCarouselList(activityId: string, limit: number = 30) {
    const list = await this.awardService.getAwardsInfoList(
      {
        activityId,
        prizeDataType: {
          $in: [PRIZE_DATA_TYPE.CARD_TYPE_AWARD, PRIZE_DATA_TYPE.JOIN_TIMES]
        }
      },
      {
        sort: {
          createTime: 1
        },
        projection: {
          name: 1,
          image: 1,
          type: 1,
          prizeDataType: 1,
          avatar: 1,
          userNick: 1
        },
        limit
      }
    )

    return list
  }
  async getMyCardsInfo(activityId: string, userInfo: IUserInfo) {
    const { cardInfo } = userInfo
    const cardPool = await this.awardService.getPrizeConfig({
      activityId,
      prizeDataType: PRIZE_DATA_TYPE.CARD
    })

    return Object.keys(cardInfo).map(type => {
      const { name, image } = cardPool.find(v => v.cardType === +type) || {}
      return {
        type,
        name,
        image,
        count: cardInfo[type] || 0
      }
    })
  }

  async getGiftCard(activityId: string, type: number, userInfo: IUserInfo, inviteUserInfo: IUserInfo) {
    // 扣除邀请者的对应卡牌
    const reduceResult = await this.userdao.update(
      {
        _id: inviteUserInfo._id,
        $where: `this.cardInfo.${type} >= 1`
      },
      {
        $inc: { [`cardInfo.${type}`]: -1 }
      }
    )

    if (!reduceResult) return CODE_TYPES.SYSTEM_ERROR

    // 增加受赠者的卡牌
    const addResult = await this.updateUser(userInfo._id, {
      $inc: { [`cardInfo.${type}`]: -1 }
    })
    if (!addResult) return CODE_TYPES.SYSTEM_ERROR

    const cardInfo =
      (await this.awardService.getPrizeConfig({
        activityId,
        prizeDataType: PRIZE_DATA_TYPE.CARD,
        cardType: +type
      })?.[0]) || {}

    const { name, image } = cardInfo

    return {
      name,
      image,
      type
    }
  }
}
