/**
 * 奖励
 *
 * @format
 */

import { BaseDao, resultsModel } from '../../sdk'
import UserService from './user.service'
import {
  CODE_TYPES,
  DRAW_STATUS,
  PRIZE_TYPE,
  ACTIVITY_STATUS,
  DELETE_STATUS,
  SWICH_TOCK,
  SHIP_STATUS
} from '../../constants'
import { AWARDS_DB_NAME, PRIZE_CONFIG_DB_NAME } from '../../db'
import { sendTBAward, getSellerSession, rand } from '../../utils'
import { generateCodeTypeWithMsg } from '../../utils/common/helper'
import { getToday } from '../../utils/common/getToday'

export default class AwardsService extends UserService {
  awardsdao: IBaseDao
  activityprizedao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.awardsdao = new BaseDao(context, AWARDS_DB_NAME)
    this.activityprizedao = new BaseDao(context, PRIZE_CONFIG_DB_NAME)
  }

  // 根据奖品Id扣库存, 扣库存成功返回为1
  async reduceStock(_id: string) {
    let result = await this.activityprizedao.findOne<IActivityPrize>({ _id, deleteStatus: ACTIVITY_STATUS.NORMAL })
    // 奖品不存在
    if (!result) {
      return CODE_TYPES.ERROR_NO_PRIZE
    }
    let { switchStock, stock, useStock } = result
    // 若不限制库存
    if (switchStock === SWICH_TOCK.NO_LIMIT) {
      return 1
    }
    // 若库存不足
    if (useStock >= stock) {
      return CODE_TYPES.ERROR_NO_STOCK
    }
    try {
      const updateResult = await this.activityprizedao.update(
        {
          _id,
          $where: 'this.useStock < this.stock'
        },
        {
          $inc: {
            useStock: +1
          }
        }
      )
      return updateResult ? 1 : CODE_TYPES.ERROR_NO_STOCK
    } catch (e) {
      console.log(`扣库存失败：`, e)
      return CODE_TYPES.SYSTEM_ERROR
    }
  }

  async getMyPrizeList(activityInfo: IActivityInfo) {
    const { openId } = this.context
    const { activityId } = this.context.data
    const myPrizeList = await this.getAwardsInfoList(
      {
        openId,
        activityId,
        type: { $ne: PRIZE_TYPE.THANKS }
      },
      {
        projection: {
          _id: 1,
          name: 1,
          image: 1,
          type: 1,
          drawStatus: 1,
          shipStatus: 1,
          receiveName: 1,
          phone: 1,
          address: 1,
          provice: 1,
          city: 1,
          area: 1,
          remark: 1,
          useUrl: 1,
          shipCompany: 1,
          shipNum: 1
        },
        sort: {
          createTime: -1
        }
      }
    )

    return {
      expiredTime: activityInfo.awardReceiveExpiredTime,
      list: this.formatMyPrizeList(myPrizeList, activityInfo.awardReceiveExpiredTime)
    }
  }

  /**
   * 插入奖品记录
   * @param {*} award
   */
  async addAward(award: IAwards) {
    return await this.awardsdao.insertOne({
      ...award,
      shipStatus: award?.type === PRIZE_TYPE.OBJECT ? SHIP_STATUS.NO_ADDRESS : '',
      createTime: Date.now(),
      updateTime: Date.now()
    })
  }

  /**
   * 发放奖品
   * @param {*} document
   */
  async sendAward(document: IAwards) {
    let { session } = await getSellerSession(this.context)
    // 发放淘宝权益
    let result = await sendTBAward(this.context, session, document)
    //@ts-ignore
    if (result.code) {
      return result
    }
    let { _id } = document

    // 更新结果
    let update = await this.awardsdao.update(
      { _id },
      {
        $set: {
          drawStatus: document.drawStatus,
          remark: document.remark,
          updateTime: Date.now()
        }
      }
    )
    console.log(`更新奖品状态`, update, document)
    return document
  }

  async getAwardsInfoList(query: object, projection?: IFindProjection) {
    return await this.awardsdao.find<IAwards>(query, projection)
  }

  formatMyPrizeList(myPrizeList: IAwards[], awardReceiveExpiredTime?: number): IAwards[] {
    const now = Date.now()
    return myPrizeList.map(v => {
      if (
        awardReceiveExpiredTime &&
        now > awardReceiveExpiredTime &&
        [DRAW_STATUS.WAITAWARD, DRAW_STATUS.RETRY].includes(v.drawStatus)
      ) {
        v.drawStatus = DRAW_STATUS.EXPIRED
        v.remark = `奖品已过期`
      }
      return {
        ...v,
        expiredTime: awardReceiveExpiredTime,
        id: v._id
      }
    })
  }

  // 领取实物
  async recieveObjectPrize(context: IContext<IParams>) {
    let { province, city, area, streetName, addressDetail, id, name, phone } = context.data

    let result = await this.awardsdao.update(
      { _id: id },
      {
        $set: {
          receiveName: name,
          phone,
          drawStatus: DRAW_STATUS.SUCCESS,
          shipStatus: SHIP_STATUS.NO_SHIP,
          province,
          city,
          area,
          streetName,
          addressDetail,
          receiveTime: Date.now(),
          updateTime: Date.now()
        }
      }
    )
    return result
      ? { province, city, area, streetName, addressDetail, id, name, phone }
      : CODE_TYPES.ERROR_RECEIVE_PRIZE
  }

  // 发放淘宝权益（奖品数据已插入到awards_info表，且状态drawStatus 为1或者6）
  async recieveEnamePrize(_id: string, awardInfo: IAwards, session: string) {
    // 发放淘宝权益
    let result = await sendTBAward(this.context, session, awardInfo)
    // 更新
    await this.awardsdao.update(
      { _id },
      {
        $set: {
          // @ts-ignore
          drawStatus: result.drawStatus,
          // @ts-ignore
          remark: result.remark,
          updateTime: Date.now()
        }
      }
    )
    if ((result as IAwards).remark) {
      return generateCodeTypeWithMsg(CODE_TYPES.SYSTEM_ERROR, (result as IAwards).remark)
    }
    return result
  }

  async drawLottery(activityId: string, prizeDataType: number, userInfo: IUserInfo) {
    const { openId } = this.context
    // 获取奖池配置
    const prizesPool = await this.getPrizeConfig({
      activityId,
      // 根据需求配置筛选条件
      prizeDataType
    })

    const thanksPrize = prizesPool.find(v => v.type === PRIZE_TYPE.THANKS) || {
      type: PRIZE_TYPE.THANKS,
      prizeDataType,
      name: '谢谢参与'
    }

    // 根据概率获取
    let prize = await this.getPrizeByProbability(prizesPool)

    // 未找到奖品,降级到谢谢参与
    if (!prize) {
      prize = thanksPrize
    }

    let reduceResult: ICodeType | number = 1

    // 不是积分奖品, 检查是否扣库存
    if (prize.type !== PRIZE_TYPE.CREDITS && prize.type !== PRIZE_TYPE.THANKS) {
      reduceResult = await this.reduceStock(prize._id)
    }

    // 扣库存失败降级到谢谢参与
    if ((reduceResult as ICodeType)?.code || !reduceResult) {
      prize = thanksPrize
    }

    const { type, _id, ename, image, name, useUrl } = prize

    const { userNick } = userInfo
    let record = {
      openId,
      prizeId: _id,
      activityId,
      drawStatus: DRAW_STATUS.WAITAWARD,
      prizeDataType,
      remark: '',
      useUrl,
      type,
      ename,
      name,
      image,
      userNick,
      createDay: getToday()
    }

    // 奖品信息 insert c_awards表
    const result = await this.addAward(record)

    return {
      id: result,
      type,
      name,
      image
    }
  }

  // 根据查询条件获取奖品配置
  async getPrizeConfig(
    query: { activityId: string; [queryParam: string]: any },
    projection: IFindProjection = {}
  ): Promise<IActivityPrize[]> {
    return this.activityprizedao.find<IActivityPrize>(
      {
        deleteStatus: DELETE_STATUS.NORMAL,
        ...query
      },
      projection
    )
  }

  // 根据概率抽取奖品
  async getPrizeByProbability(prizes: Array<IActivityPrize>): Promise<IActivityPrize> {
    // 获取 1-10000的随机数
    const probability = rand(10000)

    return prizes.find(v => probability <= v.properiodto && probability >= v.properiodfrom)
  }

  /**
   * 获取活动配置项奖品
   * @param {string} _id
   */
  async getActivityPrizeById(_id: string) {
    return await this.activityprizedao.findOne(
      { _id, deleteStatus: ACTIVITY_STATUS.NORMAL },
      {
        sort: {
          index: 1
        }
      }
    )
  }
}
