import capiFn from './capiFn';
/**
 * 时间格式化
 * @param date    接收可以被new Date()方法转换的内容
 * @param format  字符串，需要的格式例如：'yyyy/MM/dd hh:mm:ss'
 * @returns {String}
 */
export const dateFormatter = (date, format = 'yyyy/MM/dd') => {
  if (!date) return '-';

  date = new Date(
    typeof date === 'string' && isNaN(date) ? date.replace(/-/g, '/') : date
  );
  let o = {
    'M+': date.getMonth() + 1,
    'd+': date.getDate(),
    'h+': date.getHours(),
    'm+': date.getMinutes(),
    's+': date.getSeconds(),
    'q+': Math.floor((date.getMonth() + 3) / 3),
    S: date.getMilliseconds()
  };
  if (/(y+)/.test(format)) {
    format = format.replace(
      RegExp.$1,
      (date.getFullYear() + '').substr(4 - RegExp.$1.length)
    );
  }
  for (let k in o) {
    if (new RegExp('(' + k + ')').test(format)) {
      format = format.replace(
        RegExp.$1,
        RegExp.$1.length === 1 ? o[k] : ('00' + o[k]).substr(('' + o[k]).length)
      );
    }
  }
  return format;
};


/**
 * 版本比较
 * @param {string} cur 当前版本
 * @param {string} compare 比较版本
 */
export const versionCompare = (cur, compare) => {
  const _cur = cur.split('.');
  const _compare = compare.split('.');
  if (+_cur[0] < +_compare[0]) {
    return false;
  } else if ( +_cur[0] === +_compare[0] && +_cur[1] < +_compare[1] ) {
    return false;
  } else if (+_cur[0] === +_compare[0] && +_cur[1] === +_compare[1] && +_cur[2] < +_compare[2]) {
    return false;
  }
  return true;
};

/**
 * 获取request请求所需参数
 * @param {object/string} value API项配置
 */
export const getRequestParams = value => {
  if (typeof value === 'string') {
    return {
      handle: value,
      method: 'POST',
      ext: {}
    }
  } else if (typeof value === 'object') {
    const { handle, method = 'POST', dedaultParams = {}, ...ext } = value;
    return {
      handle,
      method,
      dedaultParams,
      ext: { ...ext }
    }
  } else {
    console.error('getRequestParams: 传参有误');
  }
};

/**
 * 生成API
 * @param {array} apiList API数据数组
 * @param {function} request 请求
 */
export const generateAPI = (apiList, request) => {
  const api = {};
  Object.entries(apiList).forEach(([key, value]) => {
    const { handle, method, ext, defaultParams } = getRequestParams(value);
    api[key] = params => request(handle, method,  Object.assign({}, defaultParams, params) , ext);
  });
  api.fn = capiFn(apiList, request);
  return api;
};

/**
 * 防连点
 * @param {fn} fn 函数
 * @param {wait} wait 时间
 */
export const throttleHandle = (fn, wait = 500) => {
  let lastTime = 0

  return function cb(...args) {
    let nowTime = Date.now()

    if(nowTime - lastTime > wait) {
      fn.call(this,...args)

      lastTime = nowTime
    }
  }
}