/** @format */
import { getToday } from '../getToday'
import { getTotalCompleteTask, getTodayCompleteGrowTask, updateGoodsTaskInfo } from '../task'
import { TASK_RATE_TYPE } from '../../../constants'
import { getUserOrderlist } from '../getUserOrderlist'
import { resultsModel } from '../../../sdk'
import { CODE_TYPES } from '../../../errorCode'

export default async function updateOrderGoods(
  context: IContext<IParams>,
  { userInfo, activityInfo, task }: IControllerInfos
): Promise<IPreUpdateQuery> {
  const taskType = 'orderGoods'
  const { value, itemIds, taskRateType, times } = activityInfo?.tasks?.[taskType] || {}
  const { completeTimes, taskInfo } = getTotalCompleteTask(taskType, userInfo)
  const { todayCompleteTimes } = getTodayCompleteGrowTask(taskType, task)

  if (!activityInfo?.tasks?.[taskType]) return {}
  // 永久任务且已完成
  if (taskRateType === TASK_RATE_TYPE.FOREVER && completeTimes >= times) {
    return {}
  }

  // 每日限制完成且完成次数达到限制
  if (taskRateType === TASK_RATE_TYPE.EVERYDAY && todayCompleteTimes >= times) {
    return {}
  }

  return taskRateType === TASK_RATE_TYPE.EVERYDAY ?
    await getEverydayGoods(context, taskType, itemIds, task, times, value, userInfo)
    : await getOrderGoods(taskType, itemIds, times, value, userInfo, taskInfo)

}

/**
 * 更新永久订单信息
 * @param taskType 
 * @param itemIds 
 * @param times 
 * @param value 
 * @param userInfo 
 */
async function getOrderGoods(taskType: string, itemIds: string, times: number, value: number, userInfo: IUserInfo, taskInfo: ITaskDetail[]): Promise<IPreUpdateQuery> {
  const today = getToday()
  const orderResult = await getUserOrderlist(
    this.context,
    //@ts-ignore
    userInfo.createTime || activityInfo?.startTime || Date.now(),
    Date.now()
  )

  const itemIdsArr = itemIds.split(',').map(v => +v)

  let projection = {
    $inc: {
      [`remainTimes.${taskType}`]: 0
    },
    $set: {}
  }

  let targetOrders = userInfo?.taskInfo?.[today]?.[taskType] || []
  orderResult.forEach(v => {
    // @ts-ignore
    // 商品订单包含目标商品 且orderId为新订单
    if (itemIdsArr.includes(v.itemId) && !taskInfo.some(order => order.orderId === v.orderId)) {
      if (targetOrders.length >= times) {
        return
      }
      projection.$inc[`remainTimes.${taskType}`] += +value
      targetOrders.push({
        itemId: v.itemId,
        orderId: v.orderId,
        payTime: v.payTime,
        tId: v.tId,
        createTime: Date.now()
      })
    }
    if (targetOrders?.length) {
      projection.$set[`taskInfo.${today}.${taskType}`] = targetOrders
    }
  })

  return projection
}

/**
 * 更新每日订单信息,仅适用于下单，如果是每日确认收货，需要单独处理
 * @param context 
 * @param taskType 
 * @param itemIds 
 * @param task 
 * @param times 
 * @param value 
 * @param userInfo 
 */
async function getEverydayGoods(context: IContext<IParams>, taskType: string, itemIds: string, task: ITask, times: number, value: number, userInfo: IUserInfo): Promise<IPreUpdateQuery> {
  const today = getToday()
  const TodayTime = new Date(today).getTime()
  let startTime = TodayTime > userInfo.createTime ? TodayTime : userInfo.createTime

  const orderResult = await getUserOrderlist(
    context,
    //@ts-ignore
    startTime,
    Date.now()
  )

  const itemIdsArr = itemIds.split(',').map(v => +v)

  let projection = {
    $inc: {
      [`remainTimes.${taskType}`]: 0
    },
    $set: {}
  }

  let targetOrders = task?.taskInfo?.[today]?.[taskType] || []
  let targetLength = targetOrders.length
  orderResult.forEach(v => {
    // @ts-ignore
    // 商品订单包含目标商品 且orderId为新订单
    if (itemIdsArr.includes(v.itemId) && !task?.taskInfo.some(order => order.orderId === v.orderId)) {
      if (targetOrders.length >= times) {
        return
      }
      projection.$inc[`remainTimes.${taskType}`] += +value
      targetOrders.push({
        itemId: v.itemId,
        orderId: v.orderId,
        payTime: v.payTime,
        tId: v.tId,
        createTime: Date.now()
      })
    }
  })
  if (targetOrders?.length && targetLength !== targetOrders?.length) {
    const result = await updateGoodsTaskInfo(context, today, taskType, targetOrders)
    if (result !== 1) {
      // @ts-ignore
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '订单任务更新失败')
    }

  }

  return projection
}
