/** @format */

import { getToday } from './getToday'
import { TASK_STATUS, TASK_RATE_TYPE } from '../../constants'
import { TASK_DB_NAME } from '../../db'
import { BaseDao } from '../../sdk'
import Task from '../../controller/task.controller'

/**
 * 根据任务类型获取任务完成情况
 *
 * @param {IUserInfo} userInfo
 * @param {string} taskType
 * @returns {array}
 * @memberof TaskService
 */
export function getTotalCompleteTask(
  taskType: string,
  userInfo: IUserInfo
): {
  completeTimes: number
  taskInfo: ITaskDetail[]
} {
  const { taskInfo = {} } = userInfo
  let targetTaskList = []

  Object.keys(taskInfo).forEach(day => {
    if (taskInfo[day]?.[taskType]) {
      const dayTaskInfo = Array.isArray(taskInfo[day]?.[taskType])
        ? taskInfo[day]?.[taskType]
        : [taskInfo[day]?.[taskType]]
      targetTaskList = [...targetTaskList, ...dayTaskInfo]
    }
  })

  return {
    completeTimes: targetTaskList?.length || 0,
    taskInfo: targetTaskList
  }
}

/**
 *
 * 获取任务今天完成的次数
 *
 * @param {string} taskType  任务类型
 * @param {IUserInfo} userInfo 用户信息
 * @returns {number} todayTimes  今日任务完成次数  todayTaskInfo  今日对应任务完成详情
 * @memberof TaskService
 */
export function getTodayCompleteTask(
  taskType: string,
  userInfo: IUserInfo
): {
  todayCompleteTimes: number
  todayTaskInfo: ITaskDetail[]
} {
  const today = getToday()
  return {
    todayCompleteTimes: userInfo?.taskInfo?.[today]?.[taskType]?.length || 0,
    todayTaskInfo: userInfo?.taskInfo?.[today]?.[taskType] || []
  }
}

/**
 *
 *  根据用户和任务完成情况,设置任务状态
 *
 * @param {IUserInfo} useInfo
 * @param {string} taskType
 * @param {number} taskRateType
 * @param {number} [limitTimesEverday] 每天限制次数 任务频率为每天的时候必填
 * @return {taskTatus} 1未完成 2 待领取 3 已完成
 * @memberof TaskService
 */
export function setTaskStatus(
  useInfo: IUserInfo,
  taskType: string,
  taskRateType: number,
  limitTimesEveryday: number = 1
): number {
  const waitReceive = useInfo?.remainTimes?.[taskType] || 0
  if (waitReceive) return TASK_STATUS.WAIT_RECEIVE

  // 当天的完成次数
  const { todayCompleteTimes } = getTodayCompleteTask(taskType, useInfo)

  // 总共的完成次数
  const { completeTimes } = getTotalCompleteTask(taskType, useInfo)

  switch (taskRateType) {
    case TASK_RATE_TYPE.FOREVER:
      return completeTimes > 0 ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.EVERYDAY:
      return todayCompleteTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.NOLIMIT:
      return TASK_STATUS.WAIT_DO
      break
  }
}

/**
 * 获取用户任务参与信息
 * @param context
 * @param openId
 * @param activityId
 */
export const getTask = async (context: IContext<IParams>, openId: string = '', activityId?: string): Promise<ITask> => {
  activityId = activityId ? activityId : context.data.activityId
  openId = openId ? openId : context.openId
  const taskInfodao: IBaseDao = new BaseDao(context, TASK_DB_NAME)
  const createDay = getToday()

  //查询每日任务完成记录
  const result: ITask = await taskInfodao.findOne({
    openId,
    activityId,
    createDay
  })

  //如果还没有今天的任务数据，则新增一条用户任务数据
  if (!result) {
    const insertObj = {
      openId,
      activityId,
      taskInfo: {},
      createTime: Date.now(),
      createDay: createDay
    }
    const insertResult = await taskInfodao.insertOne(insertObj)
    return { ...insertObj, _id: insertResult }
  }

  return result
}

/**
 * 获取养成类每日任务信息
 * @param taskType
 * @param task
 */
export function getTodayCompleteGrowTask(
  taskType: string,
  task: ITask
): {
  todayCompleteTimes: number
  todayTaskInfo: ITaskDetail[]
} {
  const today = getToday()
  return {
    todayCompleteTimes: task?.taskInfo?.[today]?.[taskType]?.length || 0,
    todayTaskInfo: task?.taskInfo?.[today]?.[taskType] || []
  }
}

/**
 * 更新每日任务，适用于购买商品每日任务
 * @param context
 * @param activityId
 */
export async function updateTaskInfo(context: IContext<IParams>, _id: string, taskType: string, value) {
  let taskInfodao: IBaseDao = new BaseDao(context, TASK_DB_NAME)
  const today = getToday()
  return await taskInfodao.update(
    {
      _id
    },
    {
      $push: {
        [`taskInfo.${today}.${taskType}`]: value
      }
    }
  )
}

/**
 * 增加每日任务每日任务,适用于普通每日任务
 * @param context
 * @param activityId
 */
export async function updateGoodsTaskInfo(context: IContext<IParams>, _id: string, taskType: string, value) {
  let taskInfodao: IBaseDao = new BaseDao(context, TASK_DB_NAME)
  const today = getToday()
  return await taskInfodao.update(
    {
      _id
    },
    {
      $set: {
        [`taskInfo.${today}.${taskType}`]: value
      }
    }
  )
}
