/** @format */

import { checkParams, services, registeInfos, preCheck } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonGrowTaskService, CommonBaseService } from '../service/common'
import { ITaskInfo } from '../service/common/task.service'
import {
  checkActivityTime,
  checkGrowTaskLimit,
  checkExchangeCreditsTask,
  checkRemainTimes,
  checkInviteId
} from '../utils/common/check'
import { preUpdate } from '../decorator/common'
import { updateVip, updateSignGrowTask, updateGrowOrderGoods } from '../utils/common/update'

export default class Task {
  /**
   * 任务列表
   */
  @checkParams(['activityId'])
  @registeInfos(['vipInfo', 'task'])
  @services([CommonGrowTaskService])
  @preCheck([checkActivityTime])
  @preUpdate([updateVip, updateSignGrowTask, updateGrowOrderGoods])
  async getTaskList(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [growTaskService]: [CommonGrowTaskService]
  ): Promise<IResult<{ list: ITaskInfo[] }>> {
    // 读取B端 tasks字段配置自动生成任务列表
    const result = await growTaskService.initTaskList(userInfo, activityInfo)

    return resultsModel.success({
      ...result
    })
  }

  /**
   * 完成任务
   *
   * 商品相关任务 需要传itemId参数
   * 积分兑换任务 需要传credits参数
   */
  @checkParams(['activityId', 'taskType', 'itemId?', 'credits?'])
  @registeInfos(['session', 'task'])
  @services([CommonGrowTaskService])
  @preCheck([checkActivityTime, checkGrowTaskLimit, checkExchangeCreditsTask])
  async doCompleteTask(
    context: IContext<IParams>,
    { userInfo, activityInfo, task }: IControllerInfos,
    [growTaskService]: [CommonGrowTaskService]
  ) {
    const { taskType } = context.data

    // 更新user表
    const result = await growTaskService.completeTask(taskType, activityInfo, userInfo, task)

    return resultsModel.success(result)
  }

  /**
   * 领取任务奖励
   * 领取的奖励增加的key值  如: gameTimes  前端传入
   */
  @checkParams(['activityId', 'taskType', 'rewardsKey'])
  @services([CommonGrowTaskService])
  @preCheck([checkActivityTime, checkRemainTimes])
  async receiveTaskRewards(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [growTaskService]: [CommonGrowTaskService]
  ): Promise<IResult<{ rewards: number }>> {
    const { rewardsKey } = context.data
    const { taskType } = context.data

    const rewardsResult = await growTaskService.receiveTaskRewards(taskType, rewardsKey, userInfo)

    return resultsModel.success(rewardsResult)
  }

  /**
   * 获取收藏商品列表
   */
  @checkParams(['activityId', 'itemIds'])
  @registeInfos(['session'])
  @services([CommonBaseService, CommonGrowTaskService])
  async getCollectGoodsList(
    context: IContext<IParams>,
    { session, userInfo }: IControllerInfos,
    [commonBaseService, growTaskService]: [CommonBaseService, CommonGrowTaskService]
  ) {
    const { itemIds } = context.data

    const list = await commonBaseService.getItemListByIds(itemIds, session)

    const listWithCollectStatus = await growTaskService.getItemListWithCollectStatus(list, userInfo)

    return resultsModel.success(listWithCollectStatus)
  }
}
