/**
 * 任务相关方法
 *
 * @format
 */

import UserService from './user.service'
import { BaseDao } from '../../sdk'
import { getToday, getUserOrderlist, generateVipUrl, formatVipCbUrl, setNewFollowUserData } from '../../utils'
import { TASK_RATE_TYPE, TASK_STATUS } from '../../constants'
import { TASK_DB_NAME } from '../../db'
import {
  getTotalCompleteTask,
  setGrowTaskStatus,
  updateTaskInfo,
  getTodayCompleteGrowTask,
  getOrderCount
} from '../../utils/common/task'
import { CODE_TYPES } from '../../errorCode'

export interface ITaskInfo {
  taskType?: string // 任务类型
  url?: string // 链接地址
  rewards: number // 任务奖励
  itemIds?: string // 商品配置ids 商品相关任务返回
  todayCompleteTimes?: number // 今日完成次数
  completeTimes?: number // 总共完成次数
  taskRateType: number // 任务频率
  times?: number // 任务每日限制次数
  waitReceive?: number // 待领取次数
  title: string // 任务文案
  status: number // 任务状态 1未完成 2 待领取 3 已完成
}
export default class TaskService extends UserService {
  taskInfodao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.taskInfodao = new BaseDao(context, TASK_DB_NAME)
  }

  // 根据活动tasks字段渲染任务
  async initTaskList(userInfo: IUserInfo, activityInfo: IActivityInfo, task: ITask) {
    let list = []
    const keyList = Object.keys(activityInfo?.tasks || {})
    for (let index = 0; index < keyList.length; index++) {
      // @ts-ignore
      const taskType: ITaskType = keyList[index]
      list.push(await this.initTask(task, taskType, activityInfo?.tasks?.[taskType]?.title, userInfo, activityInfo))
    }
    return {
      list
    }
  }
  /**
   *  根据任务类型初始化   b端 tasks字段必须有相应的配置
   *
   * @param {ITaskType} taskType 任务类型
   * @param {string} title 任务标题
   * @param {IActivityInfo} activityInfo 任务配置
   * @param {IUserInfo} userInfo 用户信息
   * @return {ITaskInfo}  {ITaskInfo} 任务信息
   * @memberof TaskService
   */
  async initTask(
    task: ITask,
    taskType: ITaskType,
    title: string,
    userInfo: IUserInfo,
    activityInfo: IActivityInfo
  ): Promise<ITaskInfo> {
    if (!activityInfo?.tasks?.[taskType]) {
      console.error(`活动缺少${taskType}任务配置项`)
    }
    const { value, itemIds = '', taskRateType, times = 1, link = '' } = activityInfo?.tasks?.[taskType] || {}
    const { remainTimes } = userInfo
    let { todayCompleteTimes } = getTodayCompleteGrowTask(taskType, task)
    let { completeTimes } = getTotalCompleteTask(taskType, userInfo)
    if (taskType === 'orderGoods') {
      //如果是下单任务，需要单独去根据订单类型去订单表获取下单列表
      todayCompleteTimes = completeTimes = await getOrderCount(
        this.context,
        userInfo.openId,
        activityInfo._id,
        taskRateType
      )
    }

    return {
      taskType,
      title,
      itemIds,
      rewards: value,
      taskRateType,
      times,
      url: taskType === 'member' ? generateVipUrl(formatVipCbUrl(this.context)) : link,
      todayCompleteTimes,
      completeTimes,
      status: setGrowTaskStatus(userInfo, taskType, taskRateType, times, todayCompleteTimes, completeTimes),
      waitReceive: remainTimes?.[taskType]
    }
  }

  /**
   *
   * 领取任务奖励
   *
   * @param {ITaskType} taskType
   * @param {string} awardsTargetKey
   * @param {IUserInfo} userInfo
   * @return {number} rewards
   * @memberof TaskService
   */
  async receiveTaskRewards(
    taskType: ITaskType,
    awardsTargetKey: string,
    userInfo: IUserInfo
  ): Promise<{
    rewards: number
  }> {
    const waitReceiveTimes = userInfo?.remainTimes?.[taskType]

    await this.updateUser(userInfo._id, {
      $set: {
        [`remainTimes.${taskType}`]: 0
      },
      $inc: {
        [awardsTargetKey]: waitReceiveTimes
      }
    })

    return {
      rewards: waitReceiveTimes
    }
  }

  /**
   *根据任务类型更新任务待领取次数和任务记录
   *
   * @param {string} taskType
   * @param {number} activityInfo
   * @param {IActivityInfo} userInfo
   * @returns {boolean}
   * @memberof TaskService
   */
  async completeTask(
    taskType: ITaskType,
    activityInfo: IActivityInfo,
    userInfo: IUserInfo,
    task: ITask = {},
    customRecord: Object = {}
  ) {
    const today = getToday()
    const rewards = activityInfo?.tasks?.[taskType]?.value || 0
    const { tasks } = activityInfo
    const {
      [taskType]: { taskRateType }
    } = tasks
    const {
      openId,
      data: { itemId }
    } = this.context

    const record = {
      itemId,
      openId: taskType === 'invites' ? openId : undefined,
      createTime: Date.now(),
      ...customRecord
    }
    if (taskRateType === TASK_RATE_TYPE.EVERYDAY) {
      //如果是每日任务，则更新每日任务表
      const result = await updateTaskInfo(this.context, task._id, taskType, record)
      if (result === 1) {
        const updateResult = await this.updateUser(userInfo._id, {
          $inc: {
            [`remainTimes.${taskType}`]: +rewards
          }
        })
        return updateResult === 1 ? { ok: 1 } : CODE_TYPES.ERROR_DO_TASK
      }
      return CODE_TYPES.ERROR_DO_TASK
    } else {
      //永久任务就更新永久任务表
      let updateQuery: IUpdateQuery = {
        $inc: {
          [`remainTimes.${taskType}`]: +rewards
        },
        $push: {
          [`taskInfo.${today}.${taskType}`]: { ...record, createTime: Date.now() }
        }
      }
      if (taskType === 'follow') {
        updateQuery.$set = {
          follow: setNewFollowUserData(userInfo.follow)
        }
      }
      const result = await this.updateUser(userInfo._id, updateQuery)
      return result === 1 ? { ok: 1 } : CODE_TYPES.ERROR_DO_TASK
    }
  }

  async getItemListWithCollectStatus(list: { list: ITaoBaoItems[] }, userInfo: IUserInfo) {
    const { taskInfo } = getTotalCompleteTask('collectGoods', userInfo)

    return {
      list: list.list.map(v => {
        return {
          ...v,
          // 完成列表是否含有itemId
          collected: taskInfo.some(completeItem => completeItem.itemId === v.itemId)
        }
      })
    }
  }

  /**
   * 获取用户任务参与信息
   * @param context
   * @param openId
   * @param activityId
   */
  async getTask(context: IContext<IParams>, openId: string = '', activityId?: string): Promise<ITask> {
    activityId = activityId ? activityId : context.data.activityId
    openId = openId ? openId : context.openId
    const createDay = getToday()

    //查询每日任务完成记录
    const result: ITask = await this.taskInfodao.findOne({
      openId,
      activityId,
      createDay
    })

    //如果还没有今天的任务数据，则新增一条用户任务数据
    if (!result) {
      const insertObj = {
        openId,
        activityId,
        taskInfo: {},
        createTime: Date.now(),
        createDay: createDay
      }
      const insertResult = await this.taskInfodao.insertOne(insertObj)
      return { ...insertObj, _id: insertResult }
    }

    return result
  }
}
