/**
 * 用户相关方法
 *
 * @format
 */

import { BaseDao, TBAPIS } from '../sdk'
import BaseService from './base.service'
import { USER_DB_NAME } from '../db'
import { getToday, formatUpdateUserProjection } from '../utils'
import { checkNewVip } from '../utils/package/userUpdate'
import { formatUserNick } from '../utils/package/format'
class UserService extends BaseService {
  userdao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.userdao = new BaseDao(context, USER_DB_NAME)
  }
  /**
   * @desc 获取当前打开活动的用户详情
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfo() {
    let { openId, data } = this.context
    let { activityId } = data
    let record = await this.userdao.findOne<IUserInfo>({ openId, activityId })
    return record
  }

  /**
   * @desc 根据openId获取用户详情
   * @desc 常用于助力分享码为用户openId, 被邀请人打开活动助力时需要获取邀请人的用户详情
   * @param {openId} openId
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoByOpenId(openId: string) {
    let { activityId } = this.context.data
    let record = await this.userdao.findOne<IUserInfo>({ openId, activityId })
    return record
  }

  /**
   * 初始化用户数据
   *
   * @param {IVipInfo} vipInfo
   * @param {IActivityInfo} activityInfo
   * @returns
   * @memberof UserService
   */
  async initUserData(vipInfo: IVipInfo, activityInfo: IActivityInfo) {
    const { openId, data } = this.context
    const { activityId, isFollow, avatar, inviteId, userNick } = data
    const today = getToday()
    console.log(`----------${today}----------`, today)
    const { tasks } = activityInfo
    const { follow, member } = tasks
    const user: IUserInfo = {
      activityId,
      avatar,
      inviteId,
      userNick,
      openId,
      // 待领取次数初始化, 根据需自定义添加
      remainTimes: {
        follow: isFollow ? follow.value : 0,
        member: vipInfo.isVip ? member.value : 0
      },
      member: {
        flag: !!vipInfo.isVip
      },
      follow: {
        flag: !!isFollow
      },
      login: {
        [today]: 1
      },
      taskInfo: {
        [today]: {}
      },
      createTime: Date.now(),
      createDay: today,
      updateTime: Date.now()
    }

    // 初始化 如果已关注添加任务记录
    if (isFollow) {
      user.taskInfo[today].follow = [{ createTime: Date.now() }]
    }
    // 初始化 如果已经是会员添加任务记录
    if (vipInfo.isVip) {
      user.taskInfo[today].member = [{ createTime: Date.now() }]
    }

    await this.userdao.insertOne(user)

    return user
  }

  async updateUserData(vipInfo: IVipInfo, userInfo: IUserInfo, activityInfo: IActivityInfo): Promise<IUserInfo> {
    const { data } = this.context
    const { isFollow } = data

    const today = getToday()
    let projection = {
      $set: {
        updateTime: Date.now()
      },
      $push: {},
      $inc: {
        [`login.${today}`]: 1
      }
    }

    const { tasks } = activityInfo
    const { follow, member } = tasks

    // 老用户
    // 之前进入活动未关注，现在进入关注，则视为新关注店铺用户
    const followBefore = userInfo.follow.flag
    const followNow = isFollow
    if (!followBefore && followNow && !userInfo.follow.newFollow) {
      // @ts-ignore
      projection.$set.follow = setNewFollowUserData(userInfo.follow)
      projection.$push[`taskInfo.${today}.follow`] = { createTime: Date.now() }
      projection.$inc[`remainTimes.follow`] = follow.value
    }

    // 之前进入活动非会员，现在进入会员，则视为新会员用户
    const isNewVip = checkNewVip(userInfo, vipInfo)
    if (isNewVip && !userInfo.member.newMember) {
      // @ts-ignore
      projection.$set.member = setNewVipUserData(userInfo.member)
      projection.$push[`taskInfo.${today}.member`] = { createTime: Date.now() }
      projection.$inc[`remainTimes.member`] = member.value
    }

    await this.updateUser(userInfo._id, projection)

    return {
      ...userInfo,
      // @ts-ignore
      follow: projection.$set.follow || userInfo.follow,
      // @ts-ignore
      member: projection.$set.member || userInfo.member
    }
  }

  /**
   * @desc 更新用户表
   * @param {用户的主键id} _id
   * @param {更新的对象} document
   * @returns 若更新成功，返回为1； 若更新失败，返回为 0 或系统直接报错
   */
  async updateUser(_id: string, projection: IUpdateQuery) {
    return await this.userdao.update({ _id }, formatUpdateUserProjection(projection))
  }

  // 获取排名列表
  async getRank(sortValueKey: string, sortTimeKey: string, limit: number = 200) {
    let { activityId } = this.context.data
    //获取排名
    let list = await this.userdao.find<IUserInfo>(
      {
        activityId: activityId,
        [sortValueKey]: { $gt: 0 }
      },
      {
        projection: { [sortValueKey]: 1, userNick: 1, avatar: 1, _id: 0 },
        sort: { [sortValueKey]: -1, [sortTimeKey]: 1 },
        limit
      }
    )

    return list.map((v, i) => {
      return {
        ...v,
        userNick: formatUserNick(v.userNick),
        rank: i + 1
      }
    })
  }

  /**
   *  获取用户排名
   *
   * @param {string} sortValueKey 排名字段
   * @param {number} userValue
   * @param {string} sortTimeKey
   * @return {number}  排行
   * @memberof UserService
   */
  async getMyRank(sortValueKey: string, sortTimeKey: string, userValue: number): Promise<number> {
    let { activityId } = this.context.data
    const { openId } = this.context
    let sameScoreList = await this.userdao.find<IUserInfo>(
      { [sortValueKey]: userValue, activityId },
      {
        sort: { [sortTimeKey]: 1 }
      }
    )
    let rank: number
    let gap = 0

    //说明有多个跟自己同分数的人
    for (let j = 0; j < sameScoreList.length; j++) {
      if (sameScoreList[j].openId === openId) {
        gap = j
      }
    }
    rank = await this.userdao.count({ [sortValueKey]: { $gt: userValue }, activityId })
    rank = rank + 1 + gap
    return rank
  }

  /**
   * @desc 根据用户主键id查找用户详情
   * @param {用户的主键id} _id
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoById(_id: string) {
    return await this.userdao.findOne({ _id })
  }
}

export default UserService
