/**
 * 任务相关方法
 *
 * @format
 */

import { merge } from 'lodash'
import UserService from './user.service'
import {
  getToday,
  getUserOrderlist,
  logger,
  generateVipUrl,
  formatVipCbUrl,
  sleep,
  setNewVipUserData,
  setNewFollowUserData,
  checkNewVip
} from '../utils'
import { TASK_RATE_TYPE, TASK_STATUS } from '../constants'

export interface ITaskInfo {
  taskType?: string // 任务类型
  url?: string // 链接地址
  rewards: number // 任务奖励
  itemIds?: string // 商品配置ids 商品相关任务返回
  todayCompleteTimes?: number // 今日完成次数
  completeTimes?: number // 总共完成次数
  taskRateType: number // 任务频率
  times?: number // 任务每日限制次数
  waitReceive?: number // 待领取次数
  title: string // 任务文案
  status: number // 任务状态 1未完成 2 待领取 3 已完成
}
export default class TaskService extends UserService {
  constructor(context: IContext<IParams>) {
    super(context)
  }
  /**
   *  根据任务类型初始化   b端 tasks字段必须有相应的配置
   *
   * @param {ITaskType} taskType 任务类型
   * @param {string} title 任务标题
   * @param {ITasks} tasks 任务配置
   * @param {IUserInfo} userInfo 用户信息
   * @return {ITaskInfo}  {ITaskInfo} 任务信息
   * @memberof TaskService
   */
  initTask(taskType: ITaskType, title: string, tasks: ITasks, userInfo: IUserInfo): ITaskInfo {
    const {
      [taskType]: { value, itemIds = '', taskRateType, times = 1, link = '' }
    } = tasks

    const { remainTimes } = userInfo
    const { todayCompleteTimes } = this.getTodayCompleteTask(taskType, userInfo)
    const { completeTimes } = this.getCompleteTaskByUserTaskInfo(taskType, userInfo)

    return {
      taskType,
      title,
      itemIds,
      rewards: value,
      taskRateType,
      times,
      url: taskType === 'member' ? generateVipUrl(formatVipCbUrl(this.context)) : link,
      todayCompleteTimes,
      completeTimes,
      status: this.setTaskStatus(userInfo, taskType, taskRateType, times),
      waitReceive: remainTimes[taskType]
    }
  }

  // 根据会员状态更新
  updateVipTask(userInfo: IUserInfo, vipInfo: IVipInfo, tasks: ITasks) {
    let { member } = userInfo

    const { newMember } = member
    const today = getToday()

    const isNewVip = checkNewVip(userInfo, vipInfo)

    const {
      member: { value }
    } = tasks

    let projection = {
      $set: {},
      $inc: {},
      $push: {}
    }

    if (isNewVip && !newMember) {
      member = setNewVipUserData(member)

      // @ts-ignore
      projection.$set.member = member
      projection.$inc['remainTimes.member'] = +value
      projection.$push[`taskInfo.${today}.member`] = {
        creatTime: Date.now()
      }
    }
    return projection
  }

  // 根据下单下单记录,更新状态
  async updateOrderGoodsTask(userInfo: IUserInfo, tasks: ITasks, activityStartTime: number, session?: string) {
    const taskType = 'orderGoods'
    const {
      [taskType]: { value, itemIds, taskRateType, times }
    } = tasks
    const { completeTimes, taskInfo } = this.getCompleteTaskByUserTaskInfo(taskType, userInfo)
    const { todayCompleteTimes } = this.getTodayCompleteTask(taskType, userInfo)
    // 永久任务且已完成
    if (taskRateType === TASK_RATE_TYPE.FOREVER && completeTimes) {
      return {}
    }

    // 每日限制完成且完成次数达到限制
    const today = getToday()
    if (taskRateType === TASK_RATE_TYPE.EVERYDAY && todayCompleteTimes >= times) {
      return {}
    }

    const orderResult = await getUserOrderlist(
      this.context,
      //@ts-ignore
      activityStartTime || Date.now(),
      Date.now(),
      session
    )

    const itemIdsArr = itemIds.split(',').map(v => +v)

    let projection = {
      $inc: {
        [`remainTimes.${taskType}`]: 0
      },
      $set: {}
    }

    let targetOrders = userInfo?.taskInfo?.[today]?.[taskType] || []
    orderResult.forEach(v => {
      // @ts-ignore
      // 商品订单包含目标商品 且orderId为新订单
      if (itemIdsArr.includes(v.itemId) && !taskInfo.some(order => order.orderId === v.orderId)) {
        if (taskRateType === TASK_RATE_TYPE.EVERYDAY && targetOrders.length >= times) {
          return
        }
        projection.$inc[`remainTimes.${taskType}`] += +value
        targetOrders.push({
          itemId: v.itemId,
          orderId: v.orderId,
          payTime: v.payTime,
          createTime: Date.now()
        })
      }
      if (targetOrders?.length) {
        projection.$set[`taskInfo.${today}.${taskType}`] = targetOrders
      }
    })

    return projection
  }

  // 更新签到任务数据
  updateSignTask(userInfo: IUserInfo, tasks: ITasks) {
    // 获取签到奖励
    const {
      sign: { value }
    } = tasks || {}

    const { todayCompleteTimes } = this.getTodayCompleteTask('sign', userInfo)
    const today = getToday()

    // 今天没有有完成记录
    return todayCompleteTimes === 0
      ? {
          $inc: {
            'remainTimes.sign': value
          },
          $push: {
            [`taskInfo.${today}.sign`]: { createTime: Date.now() }
          }
        }
      : {}
  }

  /**
   *更新关注店铺任务状态
   *
   * @param {IUserInfo} userInfo
   * @param {IActivityInfo} activityInfo
   * @returns
   * @memberof TaskService
   */
  async updateFollowTask(userInfo: IUserInfo, activityInfo: IActivityInfo) {
    const today = getToday()
    let { _id, follow } = userInfo
    // 获取活动基本信息
    const { tasks } = activityInfo
    // 获取关注店铺奖励
    const {
      follow: { value }
    } = tasks || {}

    // 更新follow信息
    const followInfo: IFollowInfo = setNewFollowUserData(follow)

    // 增加待领取次数
    return await this.updateUser(_id, {
      $set: {
        follow: followInfo
      },
      $inc: {
        'remainTimes.follow': value
      },
      $push: {
        [`taskInfo.${today}.follow`]: {
          creatTime: Date.now()
        }
      }
    })
  }

  /**
   *
   *  根据用户和任务完成情况,设置任务状态
   *
   * @param {IUserInfo} useInfo
   * @param {string} taskType
   * @param {number} taskRateType
   * @param {number} [limitTimesEverday] 每天限制次数 任务频率为每天的时候必填
   * @return {taskTatus} 1未完成 2 待领取 3 已完成
   * @memberof TaskService
   */
  setTaskStatus(useInfo: IUserInfo, taskType: string, taskRateType: number, limitTimesEveryday: number = 1): number {
    const waitReceive = useInfo?.remainTimes?.[taskType] || 0
    if (waitReceive) return TASK_STATUS.WAIT_RECEIVE

    // 当天的完成次数
    const { todayCompleteTimes } = this.getTodayCompleteTask(taskType, useInfo)

    // 总共的完成次数
    const { completeTimes } = this.getCompleteTaskByUserTaskInfo(taskType, useInfo)

    switch (taskRateType) {
      case TASK_RATE_TYPE.FOREVER:
        return completeTimes > 0 ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
        break
      case TASK_RATE_TYPE.EVERYDAY:
        return todayCompleteTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
        break
      case TASK_RATE_TYPE.NOLIMIT:
        return TASK_STATUS.WAIT_DO
        break
    }
  }

  /**
   *
   * 获取任务今天完成的次数
   *
   * @param {string} taskType  任务类型
   * @param {IUserInfo} userInfo 用户信息
   * @returns {number} todayTimes  今日任务完成次数  todayTaskInfo  今日对应任务完成详情
   * @memberof TaskService
   */
  getTodayCompleteTask(
    taskType: string,
    userInfo: IUserInfo
  ): {
    todayCompleteTimes: number
    todayTaskInfo: ITaskDetail[]
  } {
    const today = getToday()
    return {
      todayCompleteTimes: userInfo?.taskInfo?.[today]?.[taskType]?.length || 0,
      todayTaskInfo: userInfo?.taskInfo?.[today]?.[taskType] || []
    }
  }
  /**
   * 根据任务类型获取任务完成情况
   *
   * @param {IUserInfo} userInfo
   * @param {string} taskType
   * @returns {array}
   * @memberof TaskService
   */
  getCompleteTaskByUserTaskInfo(
    taskType: string,
    userInfo: IUserInfo
  ): {
    completeTimes: number
    taskInfo: ITaskDetail[]
  } {
    const { taskInfo = {} } = userInfo
    let targetTaskList = []

    Object.keys(taskInfo).forEach(day => {
      if (taskInfo[day]?.[taskType]) {
        const dayTaskInfo = Array.isArray(taskInfo[day]?.[taskType])
          ? taskInfo[day]?.[taskType]
          : [taskInfo[day]?.[taskType]]
        targetTaskList = [...targetTaskList, ...dayTaskInfo]
      }
    })

    return {
      completeTimes: targetTaskList?.length || 0,
      taskInfo: targetTaskList
    }
  }

  /**
   *根据任务类型更新任务待领取次数和任务记录
   *
   * @param {string} taskType
   * @param {number} rewards
   * @param {IUserInfo} userInfo
   * @param {object} [customRecord]
   * @returns {boolean}
   * @memberof TaskService
   */
  async updateUserTaskRecord(
    taskType: string,
    rewards: number,
    userInfo: IUserInfo,
    customRecord?: object
  ): Promise<boolean> {
    const { _id } = userInfo
    const today = getToday()
    const result = await this.updateUser(_id, {
      $inc: {
        [`remainTimes.${taskType}`]: rewards
      },
      $push: {
        [`taskInfo.${today}.${taskType}`]: { createTime: Date.now(), ...customRecord }
      }
    })
    return result === 1
  }

  /**
   *
   *  更新user表中,任务相关字段
   *
   * @param {string} id
   * @param {object[]} needUpdateKeysArray
   * @param {IUserInfo} userInfo
   * @returns {IUserInfo | boolean} userInfo
   * @memberof TaskService
   */
  async updateTasks(id: string, needUpdateKeysArray: object[], userInfo: IUserInfo): Promise<false | IUserInfo> {
    logger(needUpdateKeysArray)
    const updateKeys = merge({}, ...needUpdateKeysArray)

    logger(updateKeys)

    const _$inc = updateKeys.$inc || {}
    const _$set = updateKeys.$set || {}
    const _$push = updateKeys.$push || {}

    // 删除空的操作
    if (!Object.keys(_$inc).length && updateKeys.$inc) {
      delete updateKeys.$inc
    }
    if (!Object.keys(_$set).length && updateKeys.$set) {
      delete updateKeys.$set
    }
    if (!Object.keys(_$push).length && updateKeys.$push) {
      delete updateKeys.$push
    }

    if (!Object.keys(updateKeys).length) {
      return userInfo
    }

    await this.updateUser(id, updateKeys)

    // 更新数据库后 延迟10ms 执行
    await sleep(10)

    return await this.getUserInfo()
  }
}
