/**
 * 奖励
 *
 * @format
 */

import { BaseDao } from '../sdk'
import UserService from './user.service'
import {
  CODE_TYPES,
  DRAW_STATUS,
  PRIZE_TYPE,
  ACTIVITY_STATUS,
  DELETE_STATUS,
  SWICH_TOCK,
  SHIP_STATUS
} from '../constants'
import { AWARDS_DB_NAME, PRIZE_CONFIG_DB_NAME } from '../db'
import { sendTBAward, getSellerSession, rand, signFigures } from '../utils'
import { CodeType } from 'taobao-mini-sdk/lib/utils/codetypes'

export default class AwardsService extends UserService {
  awardsdao: IBaseDao
  activityprizedao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.awardsdao = new BaseDao(context, AWARDS_DB_NAME)
    this.activityprizedao = new BaseDao(context, PRIZE_CONFIG_DB_NAME)
  }

  // 根据奖品Id扣库存, 扣库存成功返回为1
  async reduceStock(_id: string) {
    let result = await this.activityprizedao.findOne<IActivityPrize>({ _id, deleteStatus: ACTIVITY_STATUS.NORMAL })
    // 奖品不存在
    if (!result) {
      return CODE_TYPES.ERROR_NO_PRIZE
    }
    let { switchStock, stock, useStock } = result
    // 若不限制库存
    if (switchStock === SWICH_TOCK.NO_LIMIT) {
      return 1
    }
    // 若库存不足
    if (useStock >= stock) {
      return CODE_TYPES.ERROR_NO_STOCK
    }
    try {
      const updateResult = await this.activityprizedao.update(
        {
          _id,
          $where: 'this.useStock < this.stock'
        },
        {
          $inc: {
            useStock: +1
          }
        }
      )
      if (updateResult !== 1) return CODE_TYPES.SYSTEM_ERROR
      return 1
    } catch (e) {
      console.log(`扣库存失败：`, e)
      return CODE_TYPES.SYSTEM_ERROR
    }
  }

  /**
   * 插入奖品记录
   * @param {*} award
   */
  async addAward(award: IAwards) {
    try {
      return await this.awardsdao.insertOne({
        ...award,
        shipStatus: award?.type === PRIZE_TYPE.OBJECT ? SHIP_STATUS.NO_ADDRESS : '',
        createTime: Date.now(),
        updateTime: Date.now()
      })
    } catch (e) {
      console.log(`插入中奖记录错误:`, e)
      return CODE_TYPES.SYSTEM_ERROR
    }
  }

  /**
   * 发放奖品
   * @param {*} document
   */
  async sendAward(document: IAwards) {
    let { session } = await getSellerSession(this.context)
    // 发放淘宝权益
    let result = await sendTBAward(this.context, session, document)
    //@ts-ignore
    if (result.code) {
      return result
    }
    let { _id } = document

    // 更新结果
    let update = await this.awardsdao.update(
      { _id },
      {
        $set: {
          drawStatus: document.drawStatus,
          remark: document.remark,
          updateTime: Date.now()
        }
      }
    )
    console.log(`更新奖品状态`, update, document)
    return document
  }

  /**
   * 我的奖品
   * @param {*} param0
   */
  async getMyPrizeList({ openId, activityId }: { openId: string; activityId: string }): Promise<Array<IAwards>> {
    // 获取奖品领取过期时间
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId)
    let myprizeList = await this.awardsdao.find<IAwards>(
      { openId, activityId, type: { $ne: PRIZE_TYPE.THANKS } },
      {
        projection: {
          _id: 1,
          name: 1,
          image: 1,
          type: 1,
          drawStatus: 1,
          shipStatus: 1,
          receiveName: 1,
          phone: 1,
          address: 1,
          provice: 1,
          city: 1,
          area: 1,
          remark: 1,
          useUrl: 1,
          shipCompany: 1,
          shipNum: 1
        },
        sort: {
          createTime: -1
        }
      }
    )

    // _id转化为id
    myprizeList = myprizeList.map(v => {
      let prize = {
        ...v,
        id: v._id
      }
      delete v._id
      return prize
    })
    // 若有过期时间，且已过期
    if (awardReceiveExpiredTime) {
      let currentTime = Date.now()
      myprizeList = myprizeList.map(v => {
        if (currentTime > awardReceiveExpiredTime) {
          if ([DRAW_STATUS.WAITAWARD, DRAW_STATUS.RETRY].includes(v.drawStatus)) {
            v.drawStatus = DRAW_STATUS.EXPIRED
            v.remark = `奖品已过期`
          }
        }
        v.expiredTime = awardReceiveExpiredTime
        return v
      })
    }
    return myprizeList
  }

  // 领取实物
  async recieveObjectPrize(
    _id: string,
    {
      activityId,
      province,
      city,
      area,
      streetName,
      addressDetail,
      name,
      phone
    }: {
      activityId: string
      province: string
      city: string
      area: string
      streetName: string
      addressDetail: string
      name: string
      phone: string | number
    }
  ) {
    // 若有过期时间，且已过期
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId)
    if (awardReceiveExpiredTime && Date.now() > awardReceiveExpiredTime) {
      return CODE_TYPES.ERROR_PRIZE_EXPIRED
    }
    let award = await this.awardsdao.findOne<IAwards>({ _id })
    console.log(`receiveObject-award`, award)
    // 奖品不存在
    if (!award) {
      return CODE_TYPES.ERROR_NO_PRIZE
    }
    // 已领取
    if (award.drawStatus === DRAW_STATUS.SUCCESS) {
      return CODE_TYPES.ERROR_RECEIVE_PRIZE
    }
    // 不是实物，非法操作
    if (award.type !== PRIZE_TYPE.OBJECT) {
      return CODE_TYPES.ERROR_FORBIDDEN_OPE
    }
    try {
      let result = await this.awardsdao.update(
        { _id },
        {
          $set: {
            receiveName: name,
            phone,
            drawStatus: DRAW_STATUS.SUCCESS,
            shipStatus: SHIP_STATUS.NO_SHIP,
            province,
            city,
            area,
            streetName,
            addressDetail,
            receiveTime: Date.now(),
            updateTime: Date.now()
          }
        }
      )
      return true
    } catch (e) {
      console.log(`领取实物错误：`, e)
      return CODE_TYPES.SYSTEM_ERROR
    }
  }

  // 发放淘宝权益（奖品数据已插入到awards_info表，且状态drawStatus 为1或者6）
  async recieveEnamePrize(_id: string, activityId: string) {
    // 若有过期时间，且已过期
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId)
    if (awardReceiveExpiredTime && Date.now() > awardReceiveExpiredTime) {
      return CODE_TYPES.ERROR_PRIZE_EXPIRED
    }
    let award = await this.awardsdao.findOne<IAwards>({ _id })
    // 奖品不存在
    if (!award) {
      return CODE_TYPES.ERROR_NO_PRIZE
    }
    // 已领取
    if (award.drawStatus === DRAW_STATUS.SUCCESS) {
      return CODE_TYPES.ERROR_RECEIVE_PRIZE
    }
    console.log(`award.drawStatus`, award.drawStatus)
    // 状态不是1,6
    if (![DRAW_STATUS.WAITAWARD, DRAW_STATUS.RETRY].includes(award.drawStatus)) {
      return CODE_TYPES.ERROR_FORBIDDEN_OPE
    }
    let { session } = await getSellerSession(this.context)
    // 发放淘宝权益
    let result = await sendTBAward(this.context, session, award)
    // 更新
    await this.awardsdao.update(
      { _id },
      {
        $set: {
          // @ts-ignore
          drawStatus: result.drawStatus,
          // @ts-ignore
          remark: result.remark,
          updateTime: Date.now()
        }
      }
    )
    return result
  }

  // 根据查询条件获取奖品配置
  async getPrizeConfig(query: { activityId: string; [queryParam: string]: any }): Promise<Array<IActivityPrize>> {
    return this.activityprizedao.find({
      deleteStatus: DELETE_STATUS.NORMAL,
      ...query
    })
  }

  // 根据概率抽取奖品
  async getPrizeByProbability(prizes: Array<IActivityPrize>): Promise<IActivityPrize> {
    // 获取 1-10000的随机数
    const probability = rand(10000)

    return prizes.find(v => probability <= v.properiodto && probability >= v.properiodfrom)
  }

  /**
   * 获取活动配置项奖品
   * @param {string} _id
   */
  async getActivityPrizeById(_id: string) {
    return await this.activityprizedao.findOne(
      { _id, deleteStatus: ACTIVITY_STATUS.NORMAL },
      {
        sort: {
          index: 1
        }
      }
    )
  }
}
