/** @format */
import { resultsModel } from '../sdk'
import { CODE_TYPES } from '../constants'
import { UserService, BaseService } from '../service'
import { getSellerSession, logFn } from '../utils'
import { getConfig } from '../config'
import { formatVipCbUrl, getShopVip } from '../utils/package/vip'
// 检验参数是否存在
export const paramsCheck = (params: string[]) => {
  return function (target: Object, name: string, descriptor: PropertyDescriptor) {
    const method = descriptor.value

    descriptor.value = function (...args: any[]) {
      const [context, otherArgs = {}] = args
      const dataKeys = Object.keys(context.data)
      for (let i = 0; i < params.length; i++) {
        const param = params[i]
        if (!dataKeys.includes(param) || context.data?.[param] === '') {
          logFn(context)
          return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少${param}参数`)
        }
      }
      return method.apply(target, [context, { ...otherArgs }])
    }
  }
}

// 检验用户是否存在
export const userCheck = ({ intercept = true }) => {
  return function (target: any, name: string, descriptor: PropertyDescriptor) {
    const method = descriptor.value
    descriptor.value = async function (...args: any[]) {
      const [context, otherArgs = {}] = args

      const userService = new UserService(context)

      const userInfo = await userService.getUserInfo()

      if (!userInfo && intercept) return resultsModel.error(CODE_TYPES.ERROR_NO_USER)

      return method.apply(target, [context, { ...otherArgs, userInfo }])
    }
  }
}

// 是否是会员
export const vipCheck = ({ intercept = true }) => {
  return function (target: any, name: string, descriptor: PropertyDescriptor) {
    const method = descriptor.value
    descriptor.value = async function (...args: any[]) {
      const [context, otherArgs = {}] = args

      const userService = new UserService(context)
      const { session, userNick } = await getSellerSession(context)

      const vipInfo = await getShopVip(context, session, formatVipCbUrl(context), userNick)
      if (!vipInfo.isVip && intercept) return resultsModel.error(CODE_TYPES.ERROR_NO_VIP, '非店铺会员', vipInfo)

      return method.apply(target, [context, { ...otherArgs, vipInfo, session }])
    }
  }
}

// 活动时间校验
export const activityTimeCheck = (target: any, name: string, descriptor: PropertyDescriptor) => {
  const method = descriptor.value
  descriptor.value = async function (...args: any[]) {
    const [context, otherArgs = {}] = args
    const { activityId } = context.data
    if (!activityId) return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少活动id`)
    const baseService = new BaseService(context)

    // 活动基本情况
    const activityInfo = await baseService.getBaseInfo(activityId)
    // 活动不存在
    if (!activityInfo) {
      return resultsModel.error(CODE_TYPES.ERROR_NO_ACTIVITY)
    }
    const { startTime, endTime } = activityInfo
    const currentTime = Date.now()
    if (currentTime < startTime) {
      return resultsModel.error(CODE_TYPES.ERROR_ACTIVITY_NOSTART, `活动未开始`)
    }
    if (currentTime > endTime) {
      return resultsModel.error(CODE_TYPES.ERROR_ACTIVITY_OVER, `活动已结束`)
    }
    return method.apply(target, [context, { ...otherArgs, activityInfo }])
  }
}
